/**
 * Alliance Living v4 - Interactive Scripts
 * Enhanced animations and smooth interactions
 */

// Global Lenis instance
let lenis = null;

document.addEventListener('DOMContentLoaded', () => {
    // Add js-active class to body for progressive enhancement
    document.body.classList.add('js-active');

    // Initialize Lenis smooth scroll FIRST
    initLenis();

    // Initialize all components
    initHeader();
    initMobileMenu();
    initHero();
    initScrollReveal();
    initCountUp();
    initSmoothScroll();
    initInstagramCarousel();
    initStackingCards();
    initTestimonialPanels();
    initCustomScrollbar();
    initParallaxImages();
    initProjectDetailSync();
    initSentenceSplit();
});


/**
 * Lenis Smooth Scroll - synced with GSAP ScrollTrigger
 */
function initLenis() {
    if (typeof Lenis === 'undefined') return;

    lenis = new Lenis({
        duration: 1.2,
        easing: (t) => Math.min(1, 1.001 - Math.pow(2, -10 * t)),  // Expo ease-out
        orientation: 'vertical',
        gestureOrientation: 'vertical',
        smoothWheel: true,
        wheelMultiplier: 1,
        touchMultiplier: 2,
        infinite: false,
        lerp: 0.1,               // Smoother interpolation for slow scrolling
    });

    // Sync Lenis with GSAP ScrollTrigger
    lenis.on('scroll', ScrollTrigger.update);

    // Use GSAP ticker for Lenis updates (better sync)
    gsap.ticker.add((time) => {
        lenis.raf(time * 1000);
    });
    gsap.ticker.lagSmoothing(0);
}

/**
 * Header scroll behavior - adds 'scrolled' class for collapsing effect
 */
function initHeader() {
    const header = document.querySelector('.site-header');
    if (!header) return;

    let ticking = false;

    const updateHeader = () => {
        if (window.scrollY > 50) {
            header.classList.add('scrolled');
        } else {
            header.classList.remove('scrolled');
        }
        ticking = false;
    };

    window.addEventListener('scroll', () => {
        if (!ticking) {
            requestAnimationFrame(updateHeader);
            ticking = true;
        }
    }, { passive: true });

    updateHeader();
}

/**
 * Mobile menu toggle
 */
function initMobileMenu() {
    const toggle = document.querySelector('.mobile-menu-toggle');
    const mobileNav = document.querySelector('.mobile-nav');
    const mobileLinks = document.querySelectorAll('.mobile-nav-link');

    if (!toggle || !mobileNav) return;

    toggle.addEventListener('click', () => {
        toggle.classList.toggle('active');
        mobileNav.classList.toggle('active');
        document.body.style.overflow = mobileNav.classList.contains('active') ? 'hidden' : '';
    });

    // Close menu when clicking a link
    mobileLinks.forEach(link => {
        link.addEventListener('click', () => {
            toggle.classList.remove('active');
            mobileNav.classList.remove('active');
            document.body.style.overflow = '';
        });
    });
}

/**
 * Hero section initialization - fully self-contained animations
 */
function initHero() {
    const hero = document.querySelector('.hero');
    if (!hero) return;

    const heroImage = hero.querySelector('.hero-image');
    const easeOutExpo = (t) => t === 1 ? 1 : 1 - Math.pow(2, -10 * t);

    // Animation configuration
    const heroAnimConfig = {
        eyebrow: { delay: 300, duration: 800 },
        titleFirst: { delay: 500, duration: 800 },
        titleLast: { delay: 650, duration: 800 },
        subtitle: { delay: 800, duration: 800 },
        cta: { delay: 950, duration: 800 },
        scrollIndicator: { delay: 1200, duration: 600 },
        heroZoom: { delay: 0, duration: 3000 }
    };

    // Animate hero image zoom + parallax (combined to prevent jump)
    if (heroImage) {
        const startTime = performance.now();
        let zoomComplete = false;

        // Combined zoom + parallax animation
        const updateTransform = () => {
            const scrolled = window.scrollY;
            const translateY = scrolled * 0.3;

            if (!zoomComplete) {
                const elapsed = performance.now() - startTime;
                const progress = Math.min(elapsed / heroAnimConfig.heroZoom.duration, 1);

                // Zoom from 1.08 to 1 while parallax is active
                const scale = 1.08 - (0.08 * progress);
                heroImage.style.setProperty('transform', `scale(${scale}) translateY(${translateY}px)`, 'important');

                if (progress < 1) {
                    requestAnimationFrame(updateTransform);
                } else {
                    zoomComplete = true;
                    heroImage.style.setProperty('animation', 'none', 'important');
                }
            } else {
                // After zoom completes, just apply parallax
                heroImage.style.setProperty('transform', `scale(1) translateY(${translateY}px)`, 'important');
            }
        };

        // Initial animation frame
        requestAnimationFrame(updateTransform);

        // Parallax scroll handler
        let ticking = false;
        window.addEventListener('scroll', () => {
            if (ticking) return;

            ticking = true;
            requestAnimationFrame(() => {
                const scrolled = window.scrollY;
                const heroHeight = hero.offsetHeight;

                if (scrolled < heroHeight) {
                    updateTransform();
                }
                ticking = false;
            });
        }, { passive: true });
    }

    // Animate hero content elements
    const animateElement = (selector, delay, duration) => {
        const elements = hero.querySelectorAll(selector);
        if (!elements.length) return;

        elements.forEach((el) => {
            // Set initial state
            el.style.setProperty('opacity', '0', 'important');
            el.style.setProperty('transform', 'translateY(20px)', 'important');
            el.style.setProperty('transition', 'none', 'important');

            // Trigger animation after delay
            setTimeout(() => {
                const startTime = performance.now();

                const animate = (currentTime) => {
                    const elapsed = currentTime - startTime;
                    const progress = Math.min(elapsed / duration, 1);
                    const eased = easeOutExpo(progress);

                    el.style.setProperty('opacity', eased, 'important');
                    // Preserve horizontal centering (translateX(-50%)) for scroll-indicator
                    const isScrollIndicator = el.classList.contains('scroll-indicator');
                    const transform = isScrollIndicator
                        ? `translateX(-50%) translateY(${20 * (1 - eased)}px)`
                        : `translateY(${20 * (1 - eased)}px)`;
                    el.style.setProperty('transform', transform, 'important');

                    if (progress < 1) {
                        requestAnimationFrame(animate);
                    }
                };

                requestAnimationFrame(animate);
            }, delay);
        });
    };

    // Animate eyebrow
    animateElement('.hero-eyebrow', heroAnimConfig.eyebrow.delay, heroAnimConfig.eyebrow.duration);

    // Animate title lines
    const titleLines = hero.querySelectorAll('.title-line');
    if (titleLines.length > 0) {
        titleLines.forEach((line, index) => {
            const delay = index === 0 ? heroAnimConfig.titleFirst.delay : heroAnimConfig.titleLast.delay;
            animateElement(`:is(${line.outerHTML})`, delay, heroAnimConfig.titleFirst.duration);
        });

        // Manual approach for title lines
        titleLines.forEach((line, index) => {
            const delay = index === 0 ? heroAnimConfig.titleFirst.delay : heroAnimConfig.titleLast.delay;
            line.style.setProperty('opacity', '0', 'important');
            line.style.setProperty('transform', 'translateY(40px)', 'important');
            line.style.setProperty('transition', 'none', 'important');

            setTimeout(() => {
                const startTime = performance.now();

                const animate = (currentTime) => {
                    const elapsed = currentTime - startTime;
                    const progress = Math.min(elapsed / heroAnimConfig.titleFirst.duration, 1);
                    const eased = easeOutExpo(progress);

                    line.style.setProperty('opacity', eased, 'important');
                    line.style.setProperty('transform', `translateY(${40 * (1 - eased)}px)`, 'important');

                    if (progress < 1) {
                        requestAnimationFrame(animate);
                    }
                };

                requestAnimationFrame(animate);
            }, delay);
        });
    }

    // Animate subtitle
    animateElement('.hero-subtitle', heroAnimConfig.subtitle.delay, heroAnimConfig.subtitle.duration);

    // Animate CTA button
    animateElement('.hero-cta', heroAnimConfig.cta.delay, heroAnimConfig.cta.duration);

    // Animate scroll indicator
    animateElement('.scroll-indicator', heroAnimConfig.scrollIndicator.delay, heroAnimConfig.scrollIndicator.duration);

    // Remove the loaded class trigger since we're handling everything in JS now
    // This prevents any CSS animations from interfering
}

/**
 * Scroll reveal animations using Intersection Observer - fully JS-based
 */
function initScrollReveal() {
    const revealElements = document.querySelectorAll('.reveal-up, .card, .certification-card, .philosophy-panel, .impact-panel');

    if (!revealElements.length) return;

    const easeOutExpo = (t) => t === 1 ? 1 : 1 - Math.pow(2, -10 * t);

    const observerOptions = {
        root: null,
        rootMargin: '0px 0px -100px 0px',
        threshold: 0.1
    };

    const observer = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
            if (entry.isIntersecting && !entry.target.dataset.revealed) {
                entry.target.dataset.revealed = 'true';
                animateReveal(entry.target);
                observer.unobserve(entry.target);
            }
        });
    }, observerOptions);

    const animateReveal = (el) => {
        const duration = 800;
        const startTime = performance.now();

        // Set initial state
        el.style.setProperty('opacity', '0', 'important');
        el.style.setProperty('transform', 'translateY(40px)', 'important');
        el.style.setProperty('transition', 'none', 'important');

        const animate = (currentTime) => {
            const elapsed = currentTime - startTime;
            const progress = Math.min(elapsed / duration, 1);
            const eased = easeOutExpo(progress);

            el.style.setProperty('opacity', eased, 'important');
            el.style.setProperty('transform', `translateY(${40 * (1 - eased)}px)`, 'important');

            if (progress < 1) {
                requestAnimationFrame(animate);
            }
        };

        requestAnimationFrame(animate);
    };

    revealElements.forEach(el => {
        observer.observe(el);
    });
}

/**
 * Animated count-up for statistics
 * Reads target number from:
 * 1. data-count attribute (if present)
 * 2. Text content inside element (parsed as number)
 * 3. Defaults to 0
 */
function initCountUp() {
    // Select only impact section numbers (exclude stacking card numbers)
    const countElements = document.querySelectorAll('[data-count], .impact-number, .stat-number');

    if (!countElements.length) return;

    const observerOptions = {
        root: null,
        rootMargin: '0px',
        threshold: 0.5
    };

    const animateCount = (element) => {
        // Skip if element is inside a service-panel or testimonial-panel
        if (element.closest('.service-panel') || element.closest('.testimonial-panel')) {
            return;
        }

        // Get target number from multiple sources
        let target = null;

        // 1. Check text content first (highest priority)
        const textContent = element.textContent.trim();
        const parsed = parseInt(textContent, 10);
        if (!isNaN(parsed)) {
            target = parsed;
        }

        // 2. Fall back to data-count attribute
        if (target === null || isNaN(target)) {
            if (element.dataset.count !== undefined) {
                target = parseInt(element.dataset.count, 10);
            }
        }

        // 3. Default to 0 if no valid number found
        if (target === null || isNaN(target)) {
            target = 0;
        }

        const duration = 2000; // 2 seconds
        const start = 0;
        const startTime = performance.now();

        const easeOutQuart = (t) => 1 - Math.pow(1 - t, 4);

        const updateCount = (currentTime) => {
            const elapsed = currentTime - startTime;
            const progress = Math.min(elapsed / duration, 1);
            const easedProgress = easeOutQuart(progress);
            const currentValue = Math.floor(start + (target - start) * easedProgress);

            element.textContent = currentValue;

            if (progress < 1) {
                requestAnimationFrame(updateCount);
            } else {
                element.textContent = target;
            }
        };

        // Only animate if we have a valid target
        if (target > 0) {
            requestAnimationFrame(updateCount);
        }
    };

    const observer = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                animateCount(entry.target);
                observer.unobserve(entry.target);
            }
        });
    }, observerOptions);

    countElements.forEach(el => observer.observe(el));
}

/**
 * Smooth scroll for anchor links - uses Lenis if available
 */
function initSmoothScroll() {
    const anchorLinks = document.querySelectorAll('a[href^="#"]');

    anchorLinks.forEach(link => {
        link.addEventListener('click', (e) => {
            const href = link.getAttribute('href');
            if (href === '#') return;

            const target = document.querySelector(href);
            if (!target) return;

            e.preventDefault();

            const headerHeight = document.querySelector('.site-header')?.offsetHeight || 0;

            // Use Lenis scrollTo if available, otherwise fallback to native
            if (lenis) {
                lenis.scrollTo(target, { offset: -headerHeight });
            } else {
                const targetPosition = target.getBoundingClientRect().top + window.scrollY - headerHeight;
                window.scrollTo({
                    top: targetPosition,
                    behavior: 'smooth'
                });
            }
        });
    });
}

/**
 * Instagram carousel functionality
 */
function initInstagramCarousel() {
    const carousel = document.querySelector('.carousel');
    if (!carousel) return;

    const track = carousel.querySelector('.carousel-track');
    const cards = carousel.querySelectorAll('.instagram-section .card');
    const prevBtn = carousel.querySelector('.carousel-btn-prev');
    const nextBtn = carousel.querySelector('.carousel-btn-next');
    let pagination = carousel.parentElement.querySelector('.carousel-pagination');

    if (!track || !cards.length) return;

    let currentIndex = 0;

    // Create pagination dots dynamically based on number of cards
    if (pagination) {
        pagination.innerHTML = '';
        cards.forEach((card, index) => {
            const dot = document.createElement('button');
            dot.className = 'carousel-dot';
            if (index === 0) dot.classList.add('active');
            dot.setAttribute('aria-label', `Go to slide ${index + 1}`);
            dot.addEventListener('click', () => {
                currentIndex = index;
                updateCarousel();
            });
            pagination.appendChild(dot);
        });
    }

    function getCardWidth() {
        const card = cards[0];
        const style = getComputedStyle(track);
        const gap = parseFloat(style.gap) || 24;
        return card.offsetWidth + gap;
    }

    function getVisibleCards() {
        const wrapperWidth = carousel.querySelector('.carousel-track-wrapper').offsetWidth;
        const cardWidth = getCardWidth();
        return Math.floor(wrapperWidth / cardWidth) || 1;
    }

    function updateCarousel() {
        const cardWidth = getCardWidth();
        const visibleCards = getVisibleCards();
        const maxIndex = Math.max(0, cards.length - visibleCards);
        currentIndex = Math.min(currentIndex, maxIndex);

        track.style.transform = `translateX(-${currentIndex * cardWidth}px)`;

        // Update button states
        if (prevBtn) prevBtn.disabled = currentIndex === 0;
        if (nextBtn) nextBtn.disabled = currentIndex >= maxIndex;

        // Update pagination dots
        if (pagination) {
            const dots = pagination.querySelectorAll('.carousel-dot');
            dots.forEach((dot, index) => {
                dot.classList.toggle('active', index === currentIndex);
            });
        }
    }

    function goToNext() {
        const visibleCards = getVisibleCards();
        const maxIndex = cards.length - visibleCards;
        if (currentIndex < maxIndex) {
            currentIndex++;
            updateCarousel();
        }
    }

    function goToPrev() {
        if (currentIndex > 0) {
            currentIndex--;
            updateCarousel();
        }
    }

    // Event listeners
    if (prevBtn) prevBtn.addEventListener('click', goToPrev);
    if (nextBtn) nextBtn.addEventListener('click', goToNext);

    // Handle resize
    let resizeTimeout;
    window.addEventListener('resize', () => {
        clearTimeout(resizeTimeout);
        resizeTimeout = setTimeout(() => {
            updateCarousel();
        }, 100);
    });

    // Touch/swipe support for mobile
    let touchStartX = 0;
    let touchEndX = 0;

    track.addEventListener('touchstart', (e) => {
        touchStartX = e.changedTouches[0].screenX;
    }, { passive: true });

    track.addEventListener('touchend', (e) => {
        touchEndX = e.changedTouches[0].screenX;
        handleSwipe();
    }, { passive: true });

    function handleSwipe() {
        const swipeThreshold = 50;
        const diff = touchStartX - touchEndX;

        if (Math.abs(diff) > swipeThreshold) {
            if (diff > 0) {
                goToNext();
            } else {
                goToPrev();
            }
        }
    }

    // Initial setup
    updateCarousel();

    // Like button functionality
    const likeButtons = carousel.querySelectorAll('.instagram-like-btn');
    likeButtons.forEach(btn => {
        btn.addEventListener('click', (e) => {
            e.stopPropagation();
            btn.classList.toggle('liked');

            // Update likes count
            const card = btn.closest('.card');
            const likesEl = card.querySelector('.instagram-likes');
            if (likesEl) {
                const currentLikes = parseInt(likesEl.textContent);
                const newLikes = btn.classList.contains('liked') ? currentLikes + 1 : currentLikes - 1;
                likesEl.textContent = `${newLikes} likes`;
            }
        });
    });

    // Prevent other buttons from triggering card click
    const actionButtons = carousel.querySelectorAll('.instagram-comment-btn, .instagram-share-btn');
    actionButtons.forEach(btn => {
        btn.addEventListener('click', (e) => {
            e.stopPropagation();
        });
    });

    // Make image clickable to open Instagram
    const images = carousel.querySelectorAll('.card-media img');
    images.forEach(img => {
        img.style.cursor = 'pointer';
        img.addEventListener('click', () => {
            window.open('https://instagram.com/alliance_living', '_blank');
        });
    });
}

/**
 * Form validation and submission handling
 */
const contactForm = document.querySelector('.contact-form');
if (contactForm) {
    contactForm.addEventListener('submit', (e) => {
        e.preventDefault();

        const submitButton = contactForm.querySelector('.form-submit');
        const originalText = submitButton.innerHTML;

        // Show loading state
        submitButton.innerHTML = '<span>Sending...</span>';
        submitButton.disabled = true;

        // Simulate form submission (replace with actual form handling)
        setTimeout(() => {
            submitButton.innerHTML = '<span>Message Sent!</span>';

            setTimeout(() => {
                submitButton.innerHTML = originalText;
                submitButton.disabled = false;
                contactForm.reset();
            }, 2000);
        }, 1500);
    });
}

/**
 * GSAP Stacking Cards Animation
 *
 * Architecture:
 * -------------
 * - Wrapper div (.services-wrapper) provides scroll runway
 * - Container (.services-container) is pinned inside wrapper
 * - Card positions are DIRECTLY mapped to scroll progress (no scrub delay)
 * - Text reveal is SCROLL-LINKED (progress-based stagger)
 * - On scroll release mid-transition, we programmatically scroll to nearest rest state
 *
 * Scroll Zones (for 3 cards):
 * ---------------------------
 * Each card has a REST zone followed by next card's TRANSITION zone
 * REST zone breakdown:
 *   - First 60%: Text reveals progressively (scroll-linked)
 *   - Last 40%: Text complete, next card starts creeping up (anticipation)
 */
function initStackingCards() {
    if (typeof gsap === 'undefined') return;
    gsap.registerPlugin(ScrollTrigger);

    const wrapper = document.querySelector(".services-wrapper");
    const container = document.querySelector(".services-container");
    const panels = gsap.utils.toArray(".service-panel");

    if (!wrapper || !container || panels.length === 0) return;

    const totalPanels = panels.length;

    // Add js-active class to enable absolute positioning for animation
    wrapper.classList.add('js-active');
    container.classList.add('js-active');

    // --- CONFIGURATION ---
    const scrollPerCard = 150; // vh per card (rest + transition combined)
    const totalScrollVh = scrollPerCard * totalPanels;
    wrapper.style.height = `${totalScrollVh}vh`;

    // --- COLLECT TEXT ELEMENTS PER PANEL ---
    const panelTextElements = panels.map(panel => {
        const number = panel.querySelector('.number-reveal');
        const words = Array.from(panel.querySelectorAll('.word-reveal .word'));
        const author = panel.querySelector('.author-reveal');
        return { number, words, author };
    });

    // --- TEXT ANIMATION HELPERS ---
    const setTextElementProgress = (el, progress) => {
        if (!el) return;
        const p = Math.max(0, Math.min(1, progress));
        el.style.opacity = p;
        el.style.filter = `blur(${8 * (1 - p)}px)`;
        el.style.transform = `translateY(${15 * (1 - p)}px)`;
    };

    const setNumberProgress = (el, progress) => {
        if (!el) return;
        const p = Math.max(0, Math.min(1, progress));
        el.style.opacity = p;
        el.style.transform = `translateX(${-30 * (1 - p)}px)`;
    };

    const setAuthorProgress = (el, progress) => {
        if (!el) return;
        const p = Math.max(0, Math.min(1, progress));
        el.style.opacity = p;
        el.style.transform = `translateY(${20 * (1 - p)}px)`;
    };

    const updatePanelText = (panelIndex, textProgress) => {
        if (panelIndex < 0 || panelIndex >= totalPanels) return;
        const { number, words, author } = panelTextElements[panelIndex];
        const t = Math.max(0, Math.min(1, textProgress));

        setNumberProgress(number, t / 0.15);

        words.forEach((word, i) => {
            const wordStart = 0.1 + (i / words.length) * 0.55;
            const wordProgress = (t - wordStart) / 0.2;
            setTextElementProgress(word, wordProgress);
        });

        setAuthorProgress(author, (t - 0.55) / 0.45);
    };

    // --- SETUP PANELS ---
    panels.forEach((panel, index) => {
        panel.style.zIndex = index + 1;
        if (index === 0) {
            gsap.set(panel, { yPercent: 0 });
        } else {
            gsap.set(panel, { yPercent: 100 });
        }
    });

    // --- CONTINUOUS SCROLL HANDLER ---
    // Each card occupies 1/totalPanels of the total progress
    // Card position is a continuous function - no discrete zones
    const updateVisuals = (progress) => {
        const progressPerCard = 1 / totalPanels;

        panels.forEach((panel, index) => {
            // Calculate this card's local progress (0 = card starts, 1 = card ends)
            const cardStart = index * progressPerCard;
            const cardEnd = (index + 1) * progressPerCard;
            const localProgress = (progress - cardStart) / progressPerCard;

            // Card Y position: First card always stays at 0, others slide in
            let yPercent;
            if (index === 0) {
                // First card always visible at yPercent 0
                yPercent = 0;
            } else if (localProgress < 0) {
                // Before this card's segment - fully hidden
                yPercent = 100;
            } else if (localProgress < 0.4) {
                // Sliding in: 100 → 0 over 0-40% of segment
                const slideProgress = localProgress / 0.4;
                yPercent = 100 - (100 * slideProgress);
            } else {
                // Resting: stays at 0
                yPercent = 0;
            }

            gsap.set(panel, { yPercent });

            // Text reveal: starts at 35% of segment, completes by 90%
            const textStart = 0.35;
            const textEnd = 0.9;
            const textProgress = (localProgress - textStart) / (textEnd - textStart);
            updatePanelText(index, textProgress);
        });
    };

    // --- CREATE SCROLL TRIGGER ---
    ScrollTrigger.create({
        id: "stackingCards",
        trigger: wrapper,
        start: "top top",
        end: "bottom bottom",
        pin: container,
        scrub: true,
        onUpdate: (self) => {
            updateVisuals(self.progress);
        }
    });
}

/**
 * GSAP Testimonial Stacking Panels Animation
 * Continuous scroll-based animation - no discrete zones
 */
function initTestimonialPanels() {
    if (typeof gsap === 'undefined') return;
    gsap.registerPlugin(ScrollTrigger);

    const wrapper = document.querySelector(".testimonials-wrapper");
    const container = document.querySelector(".testimonials-container");
    const panels = gsap.utils.toArray(".testimonial-panel");

    if (!wrapper || !container || panels.length === 0) return;

    const totalPanels = panels.length;

    // Add js-active class to enable absolute positioning for animation
    wrapper.classList.add('js-active');
    container.classList.add('js-active');

    // --- CONFIGURATION ---
    const scrollPerCard = 150; // vh per card
    const totalScrollVh = scrollPerCard * totalPanels;
    wrapper.style.height = `${totalScrollVh}vh`;

    // --- COLLECT TEXT ELEMENTS PER PANEL ---
    const panelTextElements = panels.map(panel => {
        const number = panel.querySelector('.number-reveal');
        const words = Array.from(panel.querySelectorAll('.word-reveal .word'));
        const author = panel.querySelector('.author-reveal');
        return { number, words, author };
    });

    // --- TEXT ANIMATION HELPERS ---
    const setTextElementProgress = (el, progress) => {
        if (!el) return;
        const p = Math.max(0, Math.min(1, progress));
        el.style.opacity = p;
        el.style.filter = `blur(${8 * (1 - p)}px)`;
        el.style.transform = `translateY(${15 * (1 - p)}px)`;
    };

    const setNumberProgress = (el, progress) => {
        if (!el) return;
        const p = Math.max(0, Math.min(1, progress));
        el.style.opacity = p;
        el.style.transform = `translateX(${-30 * (1 - p)}px)`;
    };

    const setAuthorProgress = (el, progress) => {
        if (!el) return;
        const p = Math.max(0, Math.min(1, progress));
        el.style.opacity = p;
        el.style.transform = `translateY(${20 * (1 - p)}px)`;
    };

    const updatePanelText = (panelIndex, textProgress) => {
        if (panelIndex < 0 || panelIndex >= totalPanels) return;
        const { number, words, author } = panelTextElements[panelIndex];
        const t = Math.max(0, Math.min(1, textProgress));

        setNumberProgress(number, t / 0.15);

        words.forEach((word, i) => {
            const wordStart = 0.1 + (i / words.length) * 0.55;
            const wordProgress = (t - wordStart) / 0.2;
            setTextElementProgress(word, wordProgress);
        });

        setAuthorProgress(author, (t - 0.55) / 0.45);
    };

    // --- SETUP PANELS ---
    panels.forEach((panel, index) => {
        panel.style.zIndex = index + 1;
        if (index === 0) {
            gsap.set(panel, { yPercent: 0 });
        } else {
            gsap.set(panel, { yPercent: 100 });
        }
    });

    // --- CONTINUOUS SCROLL HANDLER ---
    const updateVisuals = (progress) => {
        const progressPerCard = 1 / totalPanels;

        panels.forEach((panel, index) => {
            const cardStart = index * progressPerCard;
            const localProgress = (progress - cardStart) / progressPerCard;

            // Card Y position: First card always stays at 0, others slide in
            let yPercent;
            if (index === 0) {
                // First card always visible at yPercent 0
                yPercent = 0;
            } else if (localProgress < 0) {
                yPercent = 100;
            } else if (localProgress < 0.4) {
                const slideProgress = localProgress / 0.4;
                yPercent = 100 - (100 * slideProgress);
            } else {
                yPercent = 0;
            }

            gsap.set(panel, { yPercent });

            // Text reveal: starts at 35% of segment, completes by 90%
            const textStart = 0.35;
            const textEnd = 0.9;
            const textProgress = (localProgress - textStart) / (textEnd - textStart);
            updatePanelText(index, textProgress);
        });
    };

    // --- CREATE SCROLL TRIGGER ---
    ScrollTrigger.create({
        id: "testimonialCards",
        trigger: wrapper,
        start: "top top",
        end: "bottom bottom",
        pin: container,
        scrub: true,
        onUpdate: (self) => {
            updateVisuals(self.progress);
        }
    });
}

/**
 * Custom Scrollbar - Minimal pill indicator (visual only)
 */
function initCustomScrollbar() {
    // Create scrollbar elements
    const scrollbarPill = document.createElement('div');
    scrollbarPill.className = 'scrollbar-pill';
    scrollbarPill.innerHTML = `<div class="scrollbar-pill-thumb"></div>`;
    document.body.appendChild(scrollbarPill);

    const thumb = scrollbarPill.querySelector('.scrollbar-pill-thumb');
    let hideTimeout = null;

    // Calculate thumb size and position
    const updateThumb = () => {
        const windowHeight = window.innerHeight;
        const documentHeight = document.documentElement.scrollHeight;
        const scrollY = window.scrollY;

        // Thumb height proportional to visible area
        const thumbHeight = Math.max(30, (windowHeight / documentHeight) * windowHeight);
        const maxScroll = documentHeight - windowHeight;
        const scrollPercent = maxScroll > 0 ? scrollY / maxScroll : 0;
        const trackHeight = windowHeight - thumbHeight;
        const thumbTop = scrollPercent * trackHeight;

        thumb.style.height = `${thumbHeight}px`;
        thumb.style.top = `${thumbTop}px`;
    };

    // Show/hide on scroll
    const onScroll = () => {
        scrollbarPill.classList.add('visible');
        clearTimeout(hideTimeout);
        hideTimeout = setTimeout(() => {
            scrollbarPill.classList.remove('visible');
        }, 1500);
        updateThumb();
    };

    // Event listeners
    window.addEventListener('scroll', onScroll, { passive: true });

    // Initial calculation
    updateThumb();
}

/**
 * Parallax Images - Very subtle parallax effect on images
 * Uses GSAP ScrollTrigger for smooth, performant animations
 */
function initParallaxImages() {
    if (typeof gsap === 'undefined' || typeof ScrollTrigger === 'undefined') return;

    // Respect reduced motion preference
    if (window.matchMedia('(prefers-reduced-motion: reduce)').matches) return;

    // Configuration - very subtle effect
    const parallaxConfig = {
        // How much the image moves (in pixels or %)
        amount: 30, // pixels of movement
        // Scale factor to ensure image covers container during parallax
        scale: 1.1,
        // Ease for smoother motion
        ease: 'none'
    };

    // Journal card images
    const journalImages = document.querySelectorAll('.journal-section .card-media img');
    journalImages.forEach(img => {
        // Set initial scale to cover container
        gsap.set(img, { scale: parallaxConfig.scale });

        gsap.to(img, {
            yPercent: -5,
            ease: parallaxConfig.ease,
            scrollTrigger: {
                trigger: img.closest('.card'),
                start: 'top bottom',
                end: 'bottom top',
                scrub: 0.5
            }
        });
    });

    // Instagram card images
    const instagramImages = document.querySelectorAll('.instagram-section .card-media img');
    instagramImages.forEach(img => {
        gsap.set(img, { scale: parallaxConfig.scale });

        gsap.to(img, {
            yPercent: -5,
            ease: parallaxConfig.ease,
            scrollTrigger: {
                trigger: img.closest('.card'),
                start: 'top bottom',
                end: 'bottom top',
                scrub: 0.5
            }
        });
    });

    // LinkedIn card images
    const linkedinImages = document.querySelectorAll('.linkedin-section .card-media img');
    linkedinImages.forEach(img => {
        gsap.set(img, { scale: parallaxConfig.scale });

        gsap.to(img, {
            yPercent: -5,
            ease: parallaxConfig.ease,
            scrollTrigger: {
                trigger: img.closest('.card'),
                start: 'top bottom',
                end: 'bottom top',
                scrub: 0.5
            }
        });
    });

    // Service panel backgrounds - these are fullscreen so use slightly different approach
    const servicePanelBgs = document.querySelectorAll('.service-panel .panel-bg');
    servicePanelBgs.forEach(img => {
        // Don't apply parallax to panels already animated by stacking cards
        // The stacking animation handles their positioning
        // But we can add a subtle scale effect on their images
        gsap.set(img, { scale: 1.05 });

        // Subtle zoom out effect as panel is viewed
        gsap.to(img, {
            scale: 1,
            ease: 'power1.out',
            scrollTrigger: {
                trigger: img.closest('.service-panel'),
                start: 'top bottom',
                end: 'bottom top',
                scrub: 1
            }
        });
    });

    // Masonry grid images (journal page)
    const masonryImages = document.querySelectorAll('.masonry-image img');
    masonryImages.forEach(img => {
        gsap.set(img, { scale: parallaxConfig.scale });

        gsap.to(img, {
            yPercent: -5,
            ease: parallaxConfig.ease,
            scrollTrigger: {
                trigger: img.closest('.masonry-item'),
                start: 'top bottom',
                end: 'bottom top',
                scrub: 0.5
            }
        });
    });

    // Any generic parallax-image class elements
    const genericParallaxImages = document.querySelectorAll('.parallax-image');
    genericParallaxImages.forEach(img => {
        const container = img.closest('.parallax-container') || img.parentElement;

        gsap.set(img, { scale: parallaxConfig.scale });

        gsap.to(img, {
            yPercent: -5,
            ease: parallaxConfig.ease,
            scrollTrigger: {
                trigger: container,
                start: 'top bottom',
                end: 'bottom top',
                scrub: 0.5
            }
        });
    });
}

/**
 * Project Detail Synchronized Scroll
 * When the text content is taller than the gallery (or vice versa),
 * both columns scroll at different rates so they finish together.
 */
function initProjectDetailSync() {
    if (typeof gsap === 'undefined' || typeof ScrollTrigger === 'undefined') return;

    const section = document.querySelector('.project-detail-section');
    const gallery = document.querySelector('.project-gallery');
    const content = document.querySelector('.project-detail-sticky');

    if (!section || !gallery || !content) return;

    // Only apply on desktop
    const mediaQuery = window.matchMedia('(min-width: 1025px)');

    let scrollTriggerInstance = null;

    const setupSync = () => {
        // Kill existing instance if any
        if (scrollTriggerInstance) {
            scrollTriggerInstance.kill();
            scrollTriggerInstance = null;
            // Reset transforms
            gsap.set(content, { clearProps: 'all' });
        }

        if (!mediaQuery.matches) return;

        // Wait for layout to settle
        requestAnimationFrame(() => {
            const galleryHeight = gallery.scrollHeight;
            const contentHeight = content.scrollHeight;
            const viewportHeight = window.innerHeight;
            const headerOffset = 120; // Account for sticky header

            // Determine which is taller
            const galleryIsTaller = galleryHeight > contentHeight;
            const tallerHeight = Math.max(galleryHeight, contentHeight);
            const shorterHeight = Math.min(galleryHeight, contentHeight);

            // Calculate how much the shorter column needs to "travel" extra
            const heightDifference = tallerHeight - shorterHeight;

            // If there's no significant difference, don't apply effect
            if (heightDifference < 100) return;

            // Remove sticky positioning - we'll handle it with GSAP
            content.style.position = 'relative';
            content.style.top = 'auto';

            if (galleryIsTaller) {
                // Content is shorter - it needs to scroll slower (move down as we scroll)
                // The content should travel from top to (heightDifference) as we scroll through the section
                gsap.to(content, {
                    y: heightDifference,
                    ease: 'none',
                    scrollTrigger: {
                        trigger: section,
                        start: `top ${headerOffset}px`,
                        end: `bottom bottom`,
                        scrub: true,
                        onUpdate: (self) => {
                            // Store instance for cleanup
                            scrollTriggerInstance = self;
                        }
                    }
                });
            } else {
                // Gallery is shorter - content scrolls normally, this case is less common
                // but we could implement inverse logic if needed
            }

            // Store the ScrollTrigger instance
            ScrollTrigger.getAll().forEach(st => {
                if (st.vars.trigger === section) {
                    scrollTriggerInstance = st;
                }
            });
        });
    };

    // Initial setup
    setupSync();

    // Re-setup on resize
    let resizeTimeout;
    window.addEventListener('resize', () => {
        clearTimeout(resizeTimeout);
        resizeTimeout = setTimeout(setupSync, 200);
    });

    // Re-setup on media query change
    mediaQuery.addEventListener('change', setupSync);
}

/**
 * Sentence Split - Wraps each sentence in a span for individual styling
 * Only processes paragraphs with class "split-sentences"
 *
 * WordPress Usage:
 * 1. Add a paragraph block
 * 2. In block settings (right sidebar) → Advanced → Additional CSS class(es)
 * 3. Add: split-sentences
 *
 * Result HTML:
 * <p class="split-sentences">
 *   <span class="sentence sentence-1">First sentence.</span>
 *   <span class="sentence sentence-2">Second sentence.</span>
 *   <span class="sentence sentence-3">Third sentence.</span>
 * </p>
 *
 * CSS Styling Examples:
 * .split-sentences .sentence-1 { font-size: 1.2em; font-weight: bold; }
 * .split-sentences .sentence-2 { color: var(--colour-muted); }
 * .split-sentences .sentence-3 { font-style: italic; }
 */
function initSentenceSplit() {
    // Only target paragraphs with the "split-sentences" class
    const paragraphs = document.querySelectorAll('.split-sentences');

    if (!paragraphs.length) return;

    paragraphs.forEach(p => {
        // Skip if already processed
        if (p.dataset.sentenceSplit === 'true') return;

        // Get the text content
        const text = p.textContent.trim();

        // Split by sentences (. ! ?) followed by space or end of string
        const sentences = text.match(/[^.!?]+[.!?]+(?:\s|$)/g);

        if (!sentences || sentences.length === 0) return;

        // Clear the paragraph
        p.innerHTML = '';

        // Wrap each sentence in a span
        sentences.forEach((sentence, index) => {
            const trimmed = sentence.trim();
            if (!trimmed) return; // Skip empty sentences

            const span = document.createElement('span');
            span.className = `sentence sentence-${index + 1}`;
            span.textContent = trimmed;

            // Add space after sentence (except last one)
            p.appendChild(span);
            if (index < sentences.length - 1) {
                p.appendChild(document.createTextNode(' '));
            }
        });

        // Mark as processed to avoid re-processing
        p.dataset.sentenceSplit = 'true';
    });
}
