<?php
if (!defined('ABSPATH')) exit;

class VSC_Editor {
    public function __construct() {
        add_action('after_setup_theme', array($this, 'add_editor_styles'));
        add_action('enqueue_block_editor_assets', array($this, 'enqueue_editor_styles'));
    }

    /**
     * Get all CSS files from the modular structure in the correct order
     * Follows ITCSS methodology: Base → Layout → Components → Pages → Utilities
     *
     * @return array Array of CSS file paths relative to assets/css/
     */
    private function get_css_files() {
        $css_dir = VSC_PLUGIN_DIR . 'assets/css/';
        $css_files = array();

        // Define folder order (ITCSS methodology)
        $folder_order = array('base', 'layout', 'components', 'pages', 'utilities');

        foreach ($folder_order as $folder) {
            $folder_path = $css_dir . $folder;

            if (!is_dir($folder_path)) {
                continue;
            }

            // Get all CSS files in this folder
            $files = glob($folder_path . '/*.css');

            if ($files) {
                // Sort alphabetically for consistency
                sort($files);

                foreach ($files as $file) {
                    // Convert absolute path to relative path
                    $relative_path = str_replace($css_dir, '', $file);
                    $css_files[] = $relative_path;
                }
            }
        }

        return $css_files;
    }

    /**
     * Determine dependencies for a CSS file based on its layer
     *
     * @param string $file Relative path to CSS file
     * @param array $enqueued_handles Already enqueued style handles
     * @return array Array of dependency handles
     */
    private function get_dependencies($file, $enqueued_handles) {
        $deps = array();

        // Base layer files
        if (strpos($file, 'base/') === 0) {
            // variables.css has no dependencies
            // Everything else depends on variables
            if (strpos($file, 'base/variables.css') === false && in_array('vsc-editor-variables', $enqueued_handles)) {
                $deps[] = 'vsc-editor-variables';
            }
        }
        // Layout layer files depend on base layer
        elseif (strpos($file, 'layout/') === 0) {
            if (in_array('vsc-editor-reset', $enqueued_handles)) {
                $deps[] = 'vsc-editor-reset';
            }
            if (in_array('vsc-editor-typography', $enqueued_handles)) {
                $deps[] = 'vsc-editor-typography';
            }
        }
        // Components, Pages, Utilities depend on layout layer
        else {
            $layout_deps = array('vsc-editor-common', 'vsc-editor-sections', 'vsc-editor-animations');
            foreach ($layout_deps as $dep) {
                if (in_array($dep, $enqueued_handles)) {
                    $deps[] = $dep;
                }
            }
        }

        return $deps;
    }

    /**
     * Generate handle name from file path
     *
     * @param string $file Relative path to CSS file
     * @return string WordPress style handle
     */
    private function get_handle($file) {
        // Remove .css extension and convert path to handle
        // Example: base/variables.css → vsc-editor-variables
        $handle = str_replace('.css', '', $file);
        $handle = str_replace('/', '-', $handle);
        return 'vsc-editor-' . $handle;
    }

    /**
     * Enqueue all modular CSS files for the block editor
     */
    public function enqueue_editor_styles() {
        $css_base_url = VSC_PLUGIN_URL . 'assets/css/';
        $css_files = $this->get_css_files();
        $enqueued_handles = array();

        foreach ($css_files as $file) {
            $handle = $this->get_handle($file);
            $file_url = $css_base_url . $file;
            $deps = $this->get_dependencies($file, $enqueued_handles);

            wp_enqueue_style($handle, $file_url, $deps, VSC_VERSION);
            $enqueued_handles[] = $handle;
        }
    }

    /**
     * Add editor styles using add_editor_style() for classic editor compatibility
     */
    public function add_editor_styles() {
        $css_base_url = VSC_PLUGIN_URL . 'assets/css/';
        $css_files = $this->get_css_files();

        foreach ($css_files as $file) {
            $file_url = $css_base_url . $file;
            add_editor_style($file_url);
        }
    }
}
