<?php
/**
 * Plugin Name:       VS Canvas
 * Plugin URI:        https://veroscale.au/
 * Description:       File-based script enqueuing for WordPress with dependency management and page targeting
 * Version:           3.0.0
 * Author:            Sabastian Nguyen - Vero Scale
 * Author URI:        https://veroscale.au/
 * License:           GPL-2.0-or-later
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain:       vs-canvas
 * Requires at least: 5.8
 * Requires PHP:      7.4
 *
 * @package VS_Canvas
 */

if (!defined('ABSPATH')) {
    exit;
}

define('VSC_VERSION', '3.0.0');
define('VSC_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('VSC_PLUGIN_URL', plugin_dir_url(__FILE__));

// Load classes
require_once VSC_PLUGIN_DIR . 'includes/helper-minified-assets.php';
require_once VSC_PLUGIN_DIR . 'includes/class-vsc-core.php';
require_once VSC_PLUGIN_DIR . 'includes/class-vsc-frontend.php';
require_once VSC_PLUGIN_DIR . 'includes/class-vsc-editor.php';
require_once VSC_PLUGIN_DIR . 'includes/class-vsc-admin.php';

/**
 * Initialize plugin
 */
function vsc_init() {
    new VSC_Core();
    new VSC_Frontend();
    new VSC_Editor();

    if (is_admin()) {
        new VSC_Admin();
    }
}
add_action('plugins_loaded', 'vsc_init');

/**
 * Enqueue plugin CSS
 * Loads all CSS modules in the correct order (ITCSS methodology)
 * Uses minified versions automatically in production (when SCRIPT_DEBUG is false)
 */
function vsc_enqueue_styles() {
    // Helper to enqueue CSS with automatic .min.css detection
    $enqueue_css = function($handle, $path, $deps = array(), $media = 'all') {
        vsc_enqueue_style($handle, VSC_PLUGIN_URL, VSC_PLUGIN_DIR, 'assets/css/' . $path, $deps, VSC_VERSION, $media);
    };

    // 1. BASE LAYER - Foundation styles
    $enqueue_css('vsc-variables', 'base/variables.css');
    $enqueue_css('vsc-reset', 'base/reset.css', array('vsc-variables'));
    $enqueue_css('vsc-typography', 'base/typography.css', array('vsc-variables'));

    // 2. LAYOUT LAYER - Page structure
    $enqueue_css('vsc-common', 'layout/common.css', array('vsc-reset', 'vsc-typography'));
    $enqueue_css('vsc-sections', 'layout/sections.css', array('vsc-common'));
    $enqueue_css('vsc-animations', 'layout/animations.css', array('vsc-common'));
    $enqueue_css('vsc-print', 'layout/print.css', array('vsc-common'), 'print');

    // 3. COMPONENTS LAYER - Reusable UI components
    $layout_deps = array('vsc-common', 'vsc-sections', 'vsc-animations');
    $enqueue_css('vsc-scrollbar', 'components/scrollbar.css', $layout_deps);
    $enqueue_css('vsc-service-panels', 'components/service-panels.css', $layout_deps);
    $enqueue_css('vsc-header', 'components/header.css', $layout_deps);
    $enqueue_css('vsc-hero', 'components/hero.css', $layout_deps);
    $enqueue_css('vsc-trust-bar', 'components/trust-bar.css', $layout_deps);
    $enqueue_css('vsc-philosophy-strip', 'components/philosophy-strip.css', $layout_deps);
    $enqueue_css('vsc-services', 'components/services.css', $layout_deps);
    $enqueue_css('vsc-journal', 'components/journal.css', $layout_deps);
    $enqueue_css('vsc-impact', 'components/impact.css', $layout_deps);
    $enqueue_css('vsc-contact', 'components/contact.css', $layout_deps);
    $enqueue_css('vsc-footer', 'components/footer.css', $layout_deps);
    $enqueue_css('vsc-instagram-carousel', 'components/instagram-carousel.css', $layout_deps);
    $enqueue_css('vsc-linkedin', 'components/linkedin.css', $layout_deps);
    $enqueue_css('vsc-minimal-footer', 'components/minimal-footer.css', $layout_deps);
    $enqueue_css('vsc-team', 'components/team.css', $layout_deps);
    $enqueue_css('vsc-cards', 'components/cards.css', $layout_deps);
    $enqueue_css('vsc-values', 'components/values.css', $layout_deps);

    // 4. PAGES LAYER - Page-specific styles
    $enqueue_css('vsc-page-about', 'pages/about.css', $layout_deps);
    $enqueue_css('vsc-page-accolades', 'pages/accolades.css', $layout_deps);
    $enqueue_css('vsc-page-blog-post', 'pages/blog-post.css', $layout_deps);
    $enqueue_css('vsc-page-individual-project', 'pages/individual-project.css', $layout_deps);
    $enqueue_css('vsc-page-journal', 'pages/journal.css', $layout_deps);
    $enqueue_css('vsc-page-legal', 'pages/legal.css', $layout_deps);
    $enqueue_css('vsc-page-projects', 'pages/projects.css', $layout_deps);
    $enqueue_css('vsc-page-residential', 'pages/residential-commercial.css', $layout_deps);
    $enqueue_css('vsc-page-services', 'pages/services.css', $layout_deps);

    // 5. UTILITIES LAYER - Single-purpose utility classes (highest specificity)
    $component_deps = array('vsc-common', 'vsc-sections', 'vsc-animations');
    $enqueue_css('vsc-util-layout', 'utilities/layout.css', $component_deps);
    $enqueue_css('vsc-util-spacing', 'utilities/spacing.css', $component_deps);
    $enqueue_css('vsc-util-colors', 'utilities/colors.css', $component_deps);
    $enqueue_css('vsc-util-typography', 'utilities/typography.css', $component_deps);
    $enqueue_css('vsc-util-misc', 'utilities/misc.css', $component_deps);
}
add_action('wp_enqueue_scripts', 'vsc_enqueue_styles');

/**
 * Activation hook
 */
function vsc_activate() {
    // Check for old database schema and migrate
    $old_scripts = get_option('vsc_scripts');
    $old_dependencies = get_option('vsc_dependencies');

    if ($old_scripts && !empty($old_scripts)) {
        vsc_migrate_old_scripts($old_scripts);
    }

    if ($old_dependencies && !empty($old_dependencies)) {
        vsc_migrate_old_dependencies($old_dependencies);
    }

    // Initialize new schema if not exists
    if (!get_option('vsc_script_settings')) {
        update_option('vsc_script_settings', array());
    }

    if (!get_option('vsc_dependency_settings')) {
        update_option('vsc_dependency_settings', array());
    }

    // Create directory structure
    $custom_dir = VSC_PLUGIN_DIR . 'assets/scripts/custom';
    if (!file_exists($custom_dir)) {
        wp_mkdir_p($custom_dir);
    }
}
register_activation_hook(__FILE__, 'vsc_activate');

/**
 * Migrate old scripts to new schema
 */
function vsc_migrate_old_scripts($old_scripts) {
    $export_file = WP_CONTENT_DIR . '/vsc-export-scripts-' . time() . '.txt';
    $export_content = "VS Canvas - Exported Scripts\n";
    $export_content .= "Date: " . current_time('mysql') . "\n";
    $export_content .= "IMPORTANT: Copy this code into script files in assets/scripts/custom/\n";
    $export_content .= str_repeat('=', 80) . "\n\n";

    foreach ($old_scripts as $script) {
        $export_content .= "Script: " . $script['name'] . "\n";
        $export_content .= "Type: " . $script['type'] . "\n";
        $export_content .= "ID: " . $script['id'] . "\n";
        $export_content .= "Suggested file: " . sanitize_file_name($script['name']) . "." . $script['type'] . "\n";
        $export_content .= str_repeat('-', 80) . "\n";
        $export_content .= $script['code'] . "\n";
        $export_content .= str_repeat('=', 80) . "\n\n";
    }

    file_put_contents($export_file, $export_content);

    $new_settings = array();
    foreach ($old_scripts as $script) {
        $new_settings[$script['id']] = array(
            'enabled' => isset($script['enabled']) ? $script['enabled'] : false,
            'pages' => isset($script['pages']) ? $script['pages'] : array(),
        );
    }

    update_option('vsc_script_settings', $new_settings);
    update_option('vsc_scripts_old_backup', $old_scripts);
    delete_option('vsc_scripts');
    set_transient('vsc_migration_complete', $export_file, 3600);
}

/**
 * Migrate old dependencies to new schema
 */
function vsc_migrate_old_dependencies($old_dependencies) {
    $new_settings = array();

    foreach ($old_dependencies as $dep) {
        $id = $dep['id'];
        $new_settings[$id] = array(
            'enabled' => isset($dep['enabled']) ? $dep['enabled'] : false,
        );
    }

    update_option('vsc_dependency_settings', $new_settings);
    update_option('vsc_dependencies_old_backup', $old_dependencies);
    delete_option('vsc_dependencies');
}
