<?php
/**
 * VS Site Vector Admin Module
 *
 * Handles admin interface setup and shared admin functionality
 *
 * @package VS_Site_Vector
 * @since 1.2.1
 */

if (!defined('ABSPATH')) {
    exit;
}

class VSSV_Admin {

    /**
     * Constructor
     */
    public function __construct() {
        $this->init_hooks();
    }

    /**
     * Initialize admin hooks
     */
    private function init_hooks() {
        add_action('admin_init', [$this, 'admin_init']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_assets']);
        add_action('admin_notices', [$this, 'show_admin_notices']);
        add_action('admin_footer', [$this, 'add_admin_footer']);
    }

    /**
     * Admin initialization
     */
    public function admin_init() {
        // Register settings
        register_setting('vssv_admin_settings', 'vssv_admin_nonce');

        // Check plugin requirements
        $this->check_plugin_requirements();

        // Initialize help tabs
        $this->init_help_tabs();
    }

    /**
     * Enqueue admin scripts and styles
     */
    public function enqueue_admin_assets($hook) {
        // Only load on VS Site Vector pages
        if (strpos($hook, 'vs-site-vector') === false) {
            return;
        }

        // Enqueue admin CSS
        wp_enqueue_style(
            'vssv-admin',
            VSSV_PLUGIN_URL . 'assets/css/admin.css',
            [],
            VSSV_VERSION
        );

        // Enqueue admin JS
        wp_enqueue_script(
            'vssv-admin',
            VSSV_PLUGIN_URL . 'assets/js/admin.js',
            ['jquery', 'wp-util'],
            VSSV_VERSION,
            true
        );

        // Localize script with nonce and data
        wp_localize_script('vssv-admin', 'vssvAdmin', [
            'nonce' => wp_create_nonce('vssv_ajax_nonce'),
            'ajaxurl' => admin_url('admin-ajax.php'),
            'version' => VSSV_VERSION
        ]);

        // Color picker support
        wp_enqueue_style('wp-color-picker');
        wp_enqueue_script('wp-color-picker');
    }

    /**
     * Show admin notices
     */
    public function show_admin_notices() {
        // Check if emergency mode is active
        if (VSSV_Config::is_emergency_disabled()) {
            $this->render_notice(
                'warning',
                'VS Site Vector Emergency Mode',
                'Emergency disable mode is active. All optimizations are bypassed. <a href="' . esc_url(admin_url('admin.php?page=vs-site-vector-settings')) . '">Go to settings</a> to re-enable.'
            );
        }

        // Check if safe mode is active
        if (VSSV_Config::is_safe_mode()) {
            $this->render_notice(
                'info',
                'VS Site Vector Safe Mode',
                'Safe mode is active. Only essential features are running. Check settings to adjust optimization scope.'
            );
        }

        // Check for missing extensions
        $this->check_missing_extensions();

    }

    /**
     * Check for missing PHP extensions
     */
    private function check_missing_extensions() {
        $missing = [];

        // Check for Imagick
        if (!extension_loaded('imagick') && !extension_loaded('gd')) {
            $missing[] = 'Image processing (Imagick or GD)';
        }

        // Check for cURL
        if (!extension_loaded('curl')) {
            $missing[] = 'cURL';
        }

        // Check for JSON
        if (!extension_loaded('json')) {
            $missing[] = 'JSON';
        }

        if (!empty($missing)) {
            $this->render_notice(
                'error',
                'Missing PHP Extensions',
                'VS Site Vector requires the following PHP extensions: ' . implode(', ', $missing)
            );
        }
    }

    /**
     * Render an admin notice
     */
    private function render_notice($type, $title, $message, $dismissible = true) {
        $dismissible_class = $dismissible ? ' is-dismissible' : '';
        $notice_id = sanitize_title($title);
        ?>
        <div class="notice notice-<?php echo esc_attr($type); ?><?php echo esc_attr($dismissible_class); ?>" id="<?php echo esc_attr($notice_id); ?>">
            <p>
                <strong><?php echo esc_html($title); ?>:</strong><br>
                <?php echo wp_kses_post($message); ?>
            </p>
        </div>
        <?php
    }

    /**
     * Initialize help tabs for admin pages
     */
    private function init_help_tabs() {
        $screen = get_current_screen();

        if (!$screen || strpos($screen->id, 'vs-site-vector') === false) {
            return;
        }

        // Add help tab
        $screen->add_help_tab([
            'id' => 'vssv-getting-started',
            'title' => 'Getting Started',
            'content' => $this->get_help_content_getting_started()
        ]);

        // Add help sidebar
        $screen->set_help_sidebar(
            '<p><strong>For more information:</strong></p>' .
            '<p><a href="https://veroscale.au/docs/vs-site-vector/" target="_blank">Documentation</a></p>' .
            '<p><a href="https://veroscale.com/support/" target="_blank">Support</a></p>'
        );
    }

    /**
     * Get help content
     */
    private function get_help_content_getting_started() {
        return '<h3>VS Site Vector - Getting Started</h3>' .
            '<p>VS Site Vector is a comprehensive WordPress performance optimization plugin. Here are the key features:</p>' .
            '<ul>' .
            '<li><strong>Caching</strong>: Automatic page and object caching</li>' .
            '<li><strong>Asset Optimization</strong>: Minify CSS, JavaScript, and images</li>' .
            '<li><strong>Database Optimization</strong>: Clean and optimize your database</li>' .
            '<li><strong>Performance Monitoring</strong>: Real-time performance metrics</li>' .
            '<li><strong>CDN Deployment</strong>: Multi-platform content delivery network integration</li>' .
            '</ul>' .
            '<p>Start by running a full optimization in the dashboard, then configure specific modules in settings.</p>';
    }

    /**
     * Check plugin requirements
     */
    private function check_plugin_requirements() {
        // Check PHP version
        if (version_compare(PHP_VERSION, '7.4', '<')) {
            add_option('vssv_requirements_missing', ['PHP 7.4 or higher required (you have ' . PHP_VERSION . ')']);
        }

        // Check WordPress version
        if (version_compare($GLOBALS['wp_version'], '5.8', '<')) {
            add_option('vssv_requirements_missing', ['WordPress 5.8 or higher required']);
        }

        // Check write permissions
        if (!wp_mkdir_p(VSSV_CACHE_DIR)) {
            add_option('vssv_requirements_missing', ['Cannot create cache directory: ' . VSSV_CACHE_DIR]);
        }
    }

    /**
     * Add content to admin footer
     */
    public function add_admin_footer() {
        $screen = get_current_screen();

        if (!$screen || strpos($screen->id, 'vs-site-vector') === false) {
            return;
        }

        echo '<div id="vssv-admin-footer">';
        echo '<p class="vssv-footer-text">';
        echo sprintf(
            'VS Site Vector v%s | <a href="%s">Settings</a> | <a href="%s" target="_blank">Documentation</a>',
            esc_html(VSSV_VERSION),
            esc_url(admin_url('admin.php?page=vs-site-vector-settings')),
            'https://veroscale.au/docs'
        );
        echo '</p>';
        echo '</div>';
    }

    /**
     * Register admin pages (called from main plugin or dashboard)
     */
    public function register_admin_pages() {
        // Main menu item is handled by Dashboard class
        // This is for additional admin functionality
    }

    /**
     * Get admin interface version
     */
    public static function get_admin_version() {
        return VSSV_VERSION;
    }
}
