<?php
/**
 * Dashboard Class
 *
 * Handles admin menu and delegates to HTMX loader for UI rendering
 *
 * @package VS_Site_Vector
 * @since 2.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class VSSV_Dashboard {

    /**
     * HTMX Loader instance
     */
    private $htmx_loader = null;

    /**
     * Constructor
     */
    public function __construct() {
        add_action('admin_menu', [$this, 'add_dashboard_page']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_scripts']);
    }

    /**
     * Set HTMX loader reference
     */
    public function set_htmx_loader($loader) {
        $this->htmx_loader = $loader;
    }

    /**
     * Add dashboard page
     */
    public function add_dashboard_page() {
        add_menu_page(
            'VS Site Vector',
            'VS Site Vector',
            'manage_options',
            'vs-site-vector',
            [$this, 'render_dashboard'],
            'dashicons-performance',
            30
        );
    }

    /**
     * Render dashboard - delegates to HTMX loader
     */
    public function render_dashboard() {
        if (!current_user_can('manage_options')) {
            wp_die(__('You do not have sufficient permissions to access this page.'));
        }

        ?>
        <div class="wrap vs-site-vector-page">
            <?php
            if ($this->htmx_loader) {
                $this->htmx_loader->render_ui();
            } else {
                // Fallback - try global
                if (isset($GLOBALS['vssv_htmx_loader'])) {
                    $GLOBALS['vssv_htmx_loader']->render_ui();
                } else {
                    echo '<p>Error: HTMX loader not initialized.</p>';
                }
            }
            ?>
        </div>
        <?php
    }

    /**
     * Enqueue scripts
     */
    public function enqueue_scripts($hook) {
        if (strpos($hook, 'vs-site-vector') === false) {
            return;
        }

        // Load Inter font
        wp_enqueue_style('vs-inter-font', 'https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap', [], VSSV_VERSION);
    }
}
