<?php
/**
 * VS Site Vector HTMX Loader
 *
 * Loads HTMX-based modern admin UI
 *
 * @package VS_Site_Vector
 * @since 1.3.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class VSSV_HTMX_Loader {

    /**
     * Toggle handler instance
     */
    private $toggle_handler;

    /**
     * Page registry instance
     */
    private $page_registry;

    /**
     * Constructor
     */
    public function __construct() {
        $this->toggle_handler = new VSSV_Toggle_Handler();
        $this->page_registry = VSSV_Page_Registry::get_instance();

        add_action('admin_enqueue_scripts', array($this, 'enqueue_assets'), 1);

        // AJAX endpoints for HTMX
        add_action('wp_ajax_vssv_htmx_load_page', array($this, 'load_page_content'));
    }

    /**
     * Check if current page is a VS Site Vector page
     */
    private function is_vssv_page() {
        global $pagenow;

        if ($pagenow !== 'admin.php') {
            return false;
        }

        $page = isset($_GET['view']) ? sanitize_text_field($_GET['view']) : '';

        return strpos($page, 'vs-site-vector') === 0 || strpos($page, 'vssv-') === 0;
    }

    /**
     * Enqueue HTMX and related assets
     */
    public function enqueue_assets($hook) {
        if (!$this->is_vssv_page() || !current_user_can('manage_options')) {
            return;
        }

        // Check if shared HTMX is already loaded, otherwise load it
        if (!wp_script_is('vs-shared-htmx', 'enqueued')) {
            wp_enqueue_script(
                'vs-shared-htmx',
                'https://unpkg.com/htmx.org@1.9.10',
                array(),
                '1.9.10',
                true
            );
        }

        // Google Fonts - Inter
        wp_enqueue_style(
            'vssv-google-fonts',
            'https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap',
            array(),
            null
        );

        // Lucide icons
        wp_enqueue_script(
            'lucide-icons',
            'https://unpkg.com/lucide@latest',
            array(),
            null,
            true
        );

        // Design System - Tokens
        wp_enqueue_style(
            'vs-design-tokens',
            VSSV_PLUGIN_URL . 'ui/styles/vs-design-tokens.css',
            array(),
            VSSV_VERSION
        );

        // Design System - Components
        wp_enqueue_style(
            'vs-components',
            VSSV_PLUGIN_URL . 'ui/styles/vs-components.css',
            array('vs-design-tokens'),
            VSSV_VERSION
        );

        // Design System - Utilities
        wp_enqueue_style(
            'vs-utilities',
            VSSV_PLUGIN_URL . 'ui/styles/vs-utilities.css',
            array('vs-design-tokens'),
            VSSV_VERSION
        );

        // Admin CSS
        wp_enqueue_style(
            'vssv-admin',
            VSSV_PLUGIN_URL . 'ui/styles/vssv-admin.css',
            array('vs-design-tokens', 'vs-components'),
            VSSV_VERSION
        );

        // Admin JS
        wp_enqueue_script(
            'vssv-admin',
            VSSV_PLUGIN_URL . 'ui/scripts/vssv-admin.js',
            array('vs-shared-htmx', 'lucide-icons'),
            VSSV_VERSION,
            true
        );

        // Pass data to JavaScript
        wp_localize_script('vssv-admin', 'vssvData', array(
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('vssv_htmx_load_page'),
            'ajaxNonce' => wp_create_nonce('vssv_ajax_nonce'),
            'adminUrl' => admin_url(),
            'version' => VSSV_VERSION,
        ));
    }

    /**
     * Render the complete HTMX UI
     * Called from VSSV_Dashboard::render_dashboard()
     */
    public function render_ui() {
        if (!current_user_can('manage_options')) {
            return;
        }

        // Get current page
        $page_param = isset($_GET['view']) ? sanitize_text_field($_GET['view']) : 'vs-site-vector';
        $page_slug = str_replace(array('vs-site-vector-', 'vssv-'), '', $page_param);
        if (empty($page_slug) || $page_slug === 'vs-site-vector') {
            $page_slug = 'dashboard';
        }

        $pages = $this->page_registry->get_navigation_items();
        ?>
        <!-- Embeddable Modern UI -->
        <div id="vssv-modern-ui" class="vs-ui">
            <!-- Inline Header with Navigation Tabs -->
            <header class="vs-header">
                <div class="vs-header-title">
                    <h1>VS Site Vector</h1>
                </div>

                <nav class="vs-nav-tabs">
                    <?php
                    $first = true;
                    foreach ($pages as $id => $page):
                        $is_active = ($page_slug === $page['slug']) || ($first && $page_slug === 'dashboard');
                        $first = false;
                    ?>
                        <button
                            class="vs-nav-tab <?php echo $is_active ? 'active' : ''; ?>"
                            hx-get="<?php echo admin_url('admin-ajax.php?action=vssv_htmx_load_page&view=' . urlencode($page['slug']) . '&_wpnonce=' . wp_create_nonce('vssv_htmx_load_page')); ?>"
                            hx-target="#vssv-content"
                            hx-swap="innerHTML"
                            hx-push-url="?page=vs-site-vector-<?php echo esc_attr($page['slug']); ?>"
                        >
                            <?php if (!empty($page['icon'])): ?>
                                <i data-lucide="<?php echo esc_attr($page['icon']); ?>" class="vs-nav-icon"></i>
                            <?php endif; ?>
                            <span><?php echo esc_html($page['title']); ?></span>
                        </button>
                    <?php endforeach; ?>
                </nav>

                <div class="vs-header-actions">
                    <span class="vs-version">v<?php echo esc_html(VSSV_VERSION); ?></span>
                    <!-- Mobile Menu Button -->
                    <button class="vs-mobile-menu-btn" onclick="vsMobileNav.open()" aria-label="Open menu">
                        <i data-lucide="menu"></i>
                    </button>
                </div>
            </header>

            <main class="vs-main">
                <div id="vssv-content">
                    <?php $this->page_registry->render_page($page_slug); ?>
                </div>
            </main>
        </div>

        <!-- Mobile Navigation Drawer -->
        <div class="vs-mobile-overlay" onclick="vsMobileNav.close()"></div>
        <nav class="vs-mobile-nav">
            <div class="vs-mobile-nav-header">
                <h2>VS Site Vector</h2>
                <button class="vs-mobile-nav-close" onclick="vsMobileNav.close()" aria-label="Close menu">
                    <i data-lucide="x"></i>
                </button>
            </div>
            <div class="vs-mobile-nav-items">
                <?php
                $first = true;
                foreach ($pages as $id => $page):
                    $is_active = ($page_slug === $page['slug']) || ($first && $page_slug === 'dashboard');
                    $first = false;
                ?>
                    <button
                        class="vs-mobile-nav-item <?php echo $is_active ? 'active' : ''; ?>"
                        hx-get="<?php echo admin_url('admin-ajax.php?action=vssv_htmx_load_page&view=' . urlencode($page['slug']) . '&_wpnonce=' . wp_create_nonce('vssv_htmx_load_page')); ?>"
                        hx-target="#vssv-content"
                        hx-swap="innerHTML"
                        hx-push-url="?page=vs-site-vector-<?php echo esc_attr($page['slug']); ?>"
                        onclick="vsMobileNav.close(); vsMobileNav.setActive(this);"
                    >
                        <?php if (!empty($page['icon'])): ?>
                            <i data-lucide="<?php echo esc_attr($page['icon']); ?>"></i>
                        <?php endif; ?>
                        <span><?php echo esc_html($page['title']); ?></span>
                    </button>
                <?php endforeach; ?>
            </div>
            <div class="vs-mobile-nav-footer">
                <span class="vs-mobile-nav-version">v<?php echo esc_html(VSSV_VERSION); ?></span>
            </div>
        </nav>

        <!-- Toast Container -->
        <div id="vs-toast-container"></div>

        <script>
        // Mobile Navigation Controller
        if (!window.vsMobileNav) {
            window.vsMobileNav = {
                overlay: null,
                nav: null,
                init: function() {
                    this.overlay = document.querySelector('.vs-mobile-overlay');
                    this.nav = document.querySelector('.vs-mobile-nav');
                },
                open: function() {
                    if (!this.nav) this.init();
                    this.overlay.classList.add('active');
                    this.nav.classList.add('active');
                    document.body.style.overflow = 'hidden';
                },
                close: function() {
                    if (!this.nav) this.init();
                    this.overlay.classList.remove('active');
                    this.nav.classList.remove('active');
                    document.body.style.overflow = '';
                },
                setActive: function(el) {
                    document.querySelectorAll('.vs-mobile-nav-item').forEach(function(item) {
                        item.classList.remove('active');
                    });
                    el.classList.add('active');
                }
            };
        }
        </script>
        <?php
    }

    /**
     * Handle AJAX page loading
     */
    public function load_page_content() {
        check_ajax_referer('vssv_htmx_load_page', '_wpnonce');

        $page_slug = isset($_GET['view']) ? sanitize_key($_GET['view']) : 'dashboard';

        // Find page by slug
        $page_id = null;
        foreach ($this->page_registry->get_pages() as $id => $page) {
            if ($page['slug'] === $page_slug) {
                $page_id = $id;
                break;
            }
        }

        if (!$page_id) {
            $page_id = 'dashboard';
        }

        $this->page_registry->render_page($page_id);

        wp_die();
    }
}
