<?php
/**
 * VS Site Vector Page Registry
 *
 * Manages registration and rendering of pages in modern HTMX view
 *
 * @package VS_Site_Vector
 * @since 2.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class VSSV_Page_Registry {

    /**
     * Singleton instance
     */
    private static $instance = null;

    /**
     * Registered pages
     */
    private $pages = array();

    /**
     * Pages directory
     */
    private $pages_dir;

    /**
     * Get singleton instance
     */
    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        $this->pages_dir = VSSV_PLUGIN_DIR . 'admin/pages/';
        $this->register_default_pages();

        // Allow other plugins to register pages
        do_action('vssv_register_pages', $this);
    }

    /**
     * Register a page
     *
     * @param string $id Unique page identifier
     * @param array $args Page configuration
     */
    public function register_page($id, $args) {
        $defaults = array(
            'title' => '',
            'slug' => $id,
            'file' => null,        // PHP file to include
            'callback' => null,    // Or callback function
            'capability' => 'manage_options',
            'position' => 10,
            'icon' => '',
            'parent' => '',
        );

        $args = wp_parse_args($args, $defaults);
        $args['id'] = $id;

        if (empty($args['title'])) {
            return false;
        }

        // Must have either file or callback
        if (empty($args['file']) && empty($args['callback'])) {
            return false;
        }

        $this->pages[$id] = $args;

        // Sort by position
        uasort($this->pages, function($a, $b) {
            return $a['position'] - $b['position'];
        });

        return true;
    }

    /**
     * Get all registered pages
     */
    public function get_pages() {
        return apply_filters('vssv_get_pages', $this->pages);
    }

    /**
     * Get a specific page
     */
    public function get_page($id) {
        return isset($this->pages[$id]) ? $this->pages[$id] : null;
    }

    /**
     * Get navigation items (top-level pages only)
     */
    public function get_navigation_items() {
        $items = array();

        foreach ($this->pages as $id => $page) {
            if (!empty($page['parent'])) {
                continue;
            }

            if (!current_user_can($page['capability'])) {
                continue;
            }

            $items[$id] = $page;
        }

        return apply_filters('vssv_navigation_items', $items);
    }

    /**
     * Render a page by ID
     */
    public function render_page($id) {
        $page = $this->get_page($id);

        if (!$page) {
            $this->render_not_found();
            return;
        }

        if (!current_user_can($page['capability'])) {
            $this->render_access_denied();
            return;
        }

        do_action('vssv_before_render_page', $id, $page);

        // Render via file include or callback
        if (!empty($page['file'])) {
            $file_path = $this->pages_dir . $page['file'];
            if (file_exists($file_path)) {
                include $file_path;
            } else {
                $this->render_error('Page file not found: ' . $page['file']);
            }
        } elseif (is_callable($page['callback'])) {
            try {
                call_user_func($page['callback']);
            } catch (Exception $e) {
                $this->render_error($e->getMessage());
            }
        }

        do_action('vssv_after_render_page', $id, $page);
    }

    /**
     * Register default pages
     */
    private function register_default_pages() {
        // Dashboard
        $this->register_page('dashboard', array(
            'title' => 'Dashboard',
            'slug' => 'dashboard',
            'file' => 'dashboard.php',
            'position' => 10,
            'icon' => 'home',
        ));

        // Web Vitals (RUM)
        $this->register_page('web-vitals', array(
            'title' => 'Web Vitals',
            'slug' => 'web-vitals',
            'file' => 'web-vitals.php',
            'position' => 20,
            'icon' => 'activity',
        ));

        // Cache
        $this->register_page('cache', array(
            'title' => 'Cache',
            'slug' => 'cache',
            'file' => 'cache.php',
            'position' => 30,
            'icon' => 'database',
        ));

        // Optimization
        $this->register_page('optimization', array(
            'title' => 'Optimization',
            'slug' => 'optimization',
            'file' => 'optimization.php',
            'position' => 40,
            'icon' => 'zap',
        ));

        // Settings
        $this->register_page('settings', array(
            'title' => 'Settings',
            'slug' => 'settings',
            'file' => 'settings.php',
            'position' => 100,
            'icon' => 'settings',
        ));
    }

    /**
     * Render not found error
     */
    private function render_not_found() {
        ?>
        <div class="vssv-page">
            <div class="vssv-card vssv-card-error">
                <h3>Page Not Found</h3>
                <p>The requested page could not be found.</p>
                <p style="margin-top: 12px;">
                    <button
                        class="vssv-btn vssv-btn-primary"
                        hx-get="<?php echo admin_url('admin-ajax.php?action=vssv_htmx_load_page&page=dashboard&_wpnonce=' . wp_create_nonce('vssv_htmx_load_page')); ?>"
                        hx-target="#vssv-content"
                        hx-swap="innerHTML"
                        hx-push-url="?page=vs-site-vector"
                    >
                        Return to Dashboard
                    </button>
                </p>
            </div>
        </div>
        <?php
    }

    /**
     * Render access denied error
     */
    private function render_access_denied() {
        ?>
        <div class="vssv-page">
            <div class="vssv-card vssv-card-error">
                <h3>Access Denied</h3>
                <p>You do not have sufficient permissions to access this page.</p>
            </div>
        </div>
        <?php
    }

    /**
     * Render error
     */
    private function render_error($message) {
        ?>
        <div class="vssv-page">
            <div class="vssv-card vssv-card-error">
                <h3>Error</h3>
                <p>An error occurred while loading this page.</p>
                <?php if (current_user_can('manage_options')): ?>
                    <p style="font-size: 12px; color: #666; margin-top: 8px;">
                        <strong>Debug:</strong> <?php echo esc_html($message); ?>
                    </p>
                <?php endif; ?>
            </div>
        </div>
        <?php
    }
}
