<?php
/**
 * VS Site Vector Toggle Handler
 *
 * Manages switching between modern HTMX view and classic WordPress view
 *
 * @package VS_Site_Vector
 * @since 1.3.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class VSSV_Toggle_Handler {

    /**
     * Option name for storing view preference
     */
    private $option_name = 'vssv_modern_view_enabled';

    /**
     * Constructor
     */
    public function __construct() {
        add_action('wp_ajax_vssv_toggle_view', array($this, 'handle_toggle'));
    }

    /**
     * Check if modern mode is enabled for current user
     *
     * @return bool
     */
    public function is_modern_mode() {
        // Check URL parameter for forced mode
        if (isset($_GET['vssv_modern'])) {
            return $_GET['vssv_modern'] === '1';
        }

        // Check user meta first
        $user_id = get_current_user_id();
        if ($user_id) {
            $user_preference = get_user_meta($user_id, $this->option_name, true);
            if ($user_preference !== '') {
                return (bool) $user_preference;
            }
        }

        // Fall back to site option, default to enabled
        return (bool) get_option($this->option_name, true);
    }

    /**
     * Set modern mode preference
     *
     * @param bool $enabled Whether modern mode is enabled
     * @param int $user_id User ID (0 for site-wide)
     */
    public function set_modern_mode($enabled, $user_id = 0) {
        if ($user_id) {
            update_user_meta($user_id, $this->option_name, $enabled ? '1' : '0');
        } else {
            update_option($this->option_name, $enabled);
        }
    }

    /**
     * Handle AJAX toggle request
     */
    public function handle_toggle() {
        check_ajax_referer('vssv_toggle_view', '_wpnonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }

        $enable_modern = isset($_POST['enable_modern']) ? (bool) $_POST['enable_modern'] : false;
        $user_id = get_current_user_id();

        $this->set_modern_mode($enable_modern, $user_id);

        // Build redirect URL
        $redirect_url = admin_url('admin.php?page=vs-site-vector');
        if (!$enable_modern) {
            $redirect_url = add_query_arg('vssv_modern', '0', $redirect_url);
        }

        wp_send_json_success(array(
            'redirect' => $redirect_url,
            'modern_enabled' => $enable_modern,
        ));
    }

    /**
     * Get toggle button HTML for classic view
     *
     * @return string
     */
    public function get_toggle_button() {
        $nonce = wp_create_nonce('vssv_toggle_view');
        $is_modern = $this->is_modern_mode();

        ob_start();
        ?>
        <button
            type="button"
            class="button vssv-view-toggle"
            data-nonce="<?php echo esc_attr($nonce); ?>"
            data-modern="<?php echo $is_modern ? '0' : '1'; ?>"
        >
            <?php echo $is_modern ? 'Switch to Classic View' : 'Switch to Modern View'; ?>
        </button>
        <script>
        (function() {
            var btn = document.querySelector('.vssv-view-toggle');
            if (!btn) return;
            btn.addEventListener('click', function() {
                var fd = new FormData();
                fd.append('action', 'vssv_toggle_view');
                fd.append('enable_modern', this.dataset.modern);
                fd.append('_wpnonce', this.dataset.nonce);
                fetch(ajaxurl, { method: 'POST', body: fd })
                    .then(function(r) { return r.json(); })
                    .then(function(response) {
                        if (response.success) {
                            window.location.href = response.data.redirect;
                        }
                    });
            });
        })();
        </script>
        <?php
        return ob_get_clean();
    }
}
