<?php
/**
 * Web Vitals Page
 *
 * Real User Monitoring (RUM) metrics from actual visitors
 *
 * @package VS_Site_Vector
 * @since 2.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

// Get stats for different periods
$vssv = VS_Site_Vector::get_instance();
$stats_7d = $vssv->get_web_vitals_stats(7);
$stats_30d = $vssv->get_web_vitals_stats(30);

// Get recent samples
global $wpdb;
$recent_samples = $wpdb->get_results(
    "SELECT * FROM {$wpdb->prefix}vssv_web_vitals
     ORDER BY created_at DESC
     LIMIT 20",
    ARRAY_A
);

// Get device breakdown
$device_breakdown = $wpdb->get_results(
    "SELECT device, COUNT(*) as count,
            AVG(lcp) as avg_lcp,
            AVG(fid) as avg_fid,
            AVG(cls) as avg_cls
     FROM {$wpdb->prefix}vssv_web_vitals
     WHERE created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)
     GROUP BY device",
    ARRAY_A
);

// Get connection type breakdown
$connection_breakdown = $wpdb->get_results(
    "SELECT connection_type, COUNT(*) as count,
            AVG(lcp) as avg_lcp
     FROM {$wpdb->prefix}vssv_web_vitals
     WHERE created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)
     AND connection_type IS NOT NULL
     GROUP BY connection_type
     ORDER BY count DESC",
    ARRAY_A
);

// Helper function to get score class
function vssv_get_score_class($metric, $value) {
    if ($value === null) return '';

    $thresholds = [
        'lcp' => [2500, 4000],   // ms
        'fid' => [100, 300],     // ms
        'cls' => [0.1, 0.25],    // score
        'ttfb' => [800, 1800],   // ms
        'fcp' => [1800, 3000],   // ms
    ];

    if (!isset($thresholds[$metric])) return '';

    if ($value <= $thresholds[$metric][0]) return 'good';
    if ($value <= $thresholds[$metric][1]) return 'needs-improvement';
    return 'poor';
}
?>
<div class="vs-page">
    <div class="vs-page-header">
        <h1>Web Vitals</h1>
        <div class="vs-page-actions">
            <select id="vssv-vitals-period" class="vs-select">
                <option value="7">Last 7 Days</option>
                <option value="30">Last 30 Days</option>
            </select>
            <button
                class="vs-btn"
                hx-get="<?php echo admin_url('admin-ajax.php?action=vssv_htmx_load_page&page=web-vitals&_wpnonce=' . wp_create_nonce('vssv_htmx_load_page')); ?>"
                hx-target="#vssv-content"
                hx-swap="innerHTML"
            >
                <i data-lucide="refresh-cw" style="width:16px;height:16px;margin-right:6px;"></i>
                Refresh
            </button>
        </div>
    </div>

    <!-- Core Web Vitals -->
    <div class="vs-card">
        <h3>
            <i data-lucide="activity" style="width:18px;height:18px;margin-right:8px;vertical-align:middle;"></i>
            Core Web Vitals (7 Days)
        </h3>
        <div class="vssv-vitals-cards">
            <!-- LCP -->
            <div class="vssv-vital-card">
                <div class="vssv-vital-header">
                    <span class="vssv-vital-name">LCP</span>
                    <span class="vssv-vital-full">Largest Contentful Paint</span>
                </div>
                <div class="vssv-vital-value <?php echo vssv_get_score_class('lcp', $stats_7d['avg_lcp'] ?? null); ?>">
                    <?php echo $stats_7d['avg_lcp'] ? number_format($stats_7d['avg_lcp'] / 1000, 2) . 's' : '--'; ?>
                </div>
                <div class="vssv-vital-target">Target: ≤ 2.5s</div>
            </div>

            <!-- FID -->
            <div class="vssv-vital-card">
                <div class="vssv-vital-header">
                    <span class="vssv-vital-name">FID</span>
                    <span class="vssv-vital-full">First Input Delay</span>
                </div>
                <div class="vssv-vital-value <?php echo vssv_get_score_class('fid', $stats_7d['avg_fid'] ?? null); ?>">
                    <?php echo $stats_7d['avg_fid'] !== null ? number_format($stats_7d['avg_fid']) . 'ms' : '--'; ?>
                </div>
                <div class="vssv-vital-target">Target: ≤ 100ms</div>
            </div>

            <!-- CLS -->
            <div class="vssv-vital-card">
                <div class="vssv-vital-header">
                    <span class="vssv-vital-name">CLS</span>
                    <span class="vssv-vital-full">Cumulative Layout Shift</span>
                </div>
                <div class="vssv-vital-value <?php echo vssv_get_score_class('cls', $stats_7d['avg_cls'] ?? null); ?>">
                    <?php echo $stats_7d['avg_cls'] !== null ? number_format($stats_7d['avg_cls'], 3) : '--'; ?>
                </div>
                <div class="vssv-vital-target">Target: ≤ 0.1</div>
            </div>
        </div>
    </div>

    <!-- Additional Metrics -->
    <div class="vs-stats-grid vs-mt-xl">
        <div class="vs-stat-card">
            <div class="vs-stat-card-value <?php echo vssv_get_score_class('ttfb', $stats_7d['avg_ttfb'] ?? null); ?>">
                <?php echo $stats_7d['avg_ttfb'] ? number_format($stats_7d['avg_ttfb']) . 'ms' : '--'; ?>
            </div>
            <div class="vs-stat-card-label">TTFB</div>
            <div class="vssv-stat-sub">Time to First Byte</div>
        </div>
        <div class="vs-stat-card">
            <div class="vs-stat-card-value <?php echo vssv_get_score_class('fcp', $stats_7d['avg_fcp'] ?? null); ?>">
                <?php echo $stats_7d['avg_fcp'] ? number_format($stats_7d['avg_fcp'] / 1000, 2) . 's' : '--'; ?>
            </div>
            <div class="vs-stat-card-label">FCP</div>
            <div class="vssv-stat-sub">First Contentful Paint</div>
        </div>
        <div class="vs-stat-card">
            <div class="vs-stat-card-value"><?php echo number_format($stats_7d['total_samples'] ?? 0); ?></div>
            <div class="vs-stat-card-label">Samples</div>
            <div class="vssv-stat-sub">Last 7 days</div>
        </div>
        <div class="vs-stat-card">
            <div class="vs-stat-card-value"><?php echo number_format($stats_7d['unique_urls'] ?? 0); ?></div>
            <div class="vs-stat-card-label">Pages</div>
            <div class="vssv-stat-sub">Unique URLs tracked</div>
        </div>
    </div>

    <!-- Device Breakdown -->
    <?php if (!empty($device_breakdown)): ?>
    <div class="vs-card vs-mt-xl">
        <h3>
            <i data-lucide="smartphone" style="width:18px;height:18px;margin-right:8px;vertical-align:middle;"></i>
            Performance by Device
        </h3>
        <table class="vs-table">
            <thead>
                <tr>
                    <th>Device</th>
                    <th>Samples</th>
                    <th>Avg LCP</th>
                    <th>Avg FID</th>
                    <th>Avg CLS</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($device_breakdown as $device): ?>
                <tr>
                    <td>
                        <i data-lucide="<?php echo $device['device'] === 'mobile' ? 'smartphone' : ($device['device'] === 'tablet' ? 'tablet' : 'monitor'); ?>"
                           style="width:14px;height:14px;margin-right:6px;vertical-align:middle;opacity:0.7;"></i>
                        <?php echo esc_html(ucfirst($device['device'] ?: 'Unknown')); ?>
                    </td>
                    <td><?php echo number_format($device['count']); ?></td>
                    <td class="<?php echo vssv_get_score_class('lcp', $device['avg_lcp']); ?>">
                        <?php echo $device['avg_lcp'] ? number_format($device['avg_lcp'] / 1000, 2) . 's' : '--'; ?>
                    </td>
                    <td class="<?php echo vssv_get_score_class('fid', $device['avg_fid']); ?>">
                        <?php echo $device['avg_fid'] !== null ? number_format($device['avg_fid']) . 'ms' : '--'; ?>
                    </td>
                    <td class="<?php echo vssv_get_score_class('cls', $device['avg_cls']); ?>">
                        <?php echo $device['avg_cls'] !== null ? number_format($device['avg_cls'], 3) : '--'; ?>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>
    <?php endif; ?>

    <!-- Connection Type Breakdown -->
    <?php if (!empty($connection_breakdown)): ?>
    <div class="vs-card vs-mt-xl">
        <h3>
            <i data-lucide="wifi" style="width:18px;height:18px;margin-right:8px;vertical-align:middle;"></i>
            Performance by Connection
        </h3>
        <table class="vs-table">
            <thead>
                <tr>
                    <th>Connection</th>
                    <th>Samples</th>
                    <th>Avg LCP</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($connection_breakdown as $conn): ?>
                <tr>
                    <td><?php echo esc_html(ucfirst($conn['connection_type'] ?: 'Unknown')); ?></td>
                    <td><?php echo number_format($conn['count']); ?></td>
                    <td class="<?php echo vssv_get_score_class('lcp', $conn['avg_lcp']); ?>">
                        <?php echo $conn['avg_lcp'] ? number_format($conn['avg_lcp'] / 1000, 2) . 's' : '--'; ?>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>
    <?php endif; ?>

    <!-- Recent Samples -->
    <?php if (!empty($recent_samples)): ?>
    <div class="vs-card vs-mt-xl">
        <h3>
            <i data-lucide="list" style="width:18px;height:18px;margin-right:8px;vertical-align:middle;"></i>
            Recent Samples
        </h3>
        <div style="overflow-x: auto;">
            <table class="vs-table">
                <thead>
                    <tr>
                        <th>URL</th>
                        <th>Device</th>
                        <th>LCP</th>
                        <th>FID</th>
                        <th>CLS</th>
                        <th>Time</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($recent_samples as $sample): ?>
                    <tr>
                        <td style="max-width:200px;overflow:hidden;text-overflow:ellipsis;white-space:nowrap;" title="<?php echo esc_attr($sample['url']); ?>">
                            <?php echo esc_html($sample['url']); ?>
                        </td>
                        <td><?php echo esc_html(ucfirst($sample['device'] ?: '-')); ?></td>
                        <td class="<?php echo vssv_get_score_class('lcp', $sample['lcp']); ?>">
                            <?php echo $sample['lcp'] ? number_format($sample['lcp'] / 1000, 2) . 's' : '-'; ?>
                        </td>
                        <td class="<?php echo vssv_get_score_class('fid', $sample['fid']); ?>">
                            <?php echo $sample['fid'] !== null ? $sample['fid'] . 'ms' : '-'; ?>
                        </td>
                        <td class="<?php echo vssv_get_score_class('cls', $sample['cls']); ?>">
                            <?php echo $sample['cls'] !== null ? number_format($sample['cls'], 3) : '-'; ?>
                        </td>
                        <td><?php echo esc_html(human_time_diff(strtotime($sample['created_at']), current_time('timestamp')) . ' ago'); ?></td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>
    <?php endif; ?>
</div>
