/**
 * VS Site Vector Diagnostic Test Suite
 * Comprehensive testing for VS Site Vector performance optimization
 *
 * @package VS_Site_Vector
 * @version 1.0.0
 * @since 1.2.1
 *
 * Usage: Add ?vssv_debug=1 to URL to enable diagnostics
 */

(function() {
    'use strict';

    // Only run diagnostics when explicitly requested
    if (!window.location.search.includes('vssv_debug=1')) {
        return;
    }

    const $ = window.jQuery;

    /**
     * Color-coded console output
     */
    const DiagnosticConsole = {
        red: (msg, details) => {
            // Removed console.log
        },
        yellow: (msg, details) => {
            // Removed console.log
        },
        green: (msg, details) => {
            // Removed console.log
        },
        blue: (msg, details) => {
            // Removed console.log
        },
        purple: (msg) => {
            // Removed console.log
        },
        cyan: (msg, details) => {
            // Removed console.log
        },
        separator: () => {
            // Removed console.log
        }
    };

    /**
     * Test result tracking
     */
    const TestResults = {
        passed: 0,
        failed: 0,
        warnings: 0,
        issues: [],
        fixes: [],
        metrics: {}
    };

    /**
     * Log test result
     */
    function logTest(testName, status, message, details = null) {
        const icon = status === 'pass' ? '✅' : status === 'fail' ? '❌' : '⚠️';

        if (status === 'pass') {
            TestResults.passed++;
            DiagnosticConsole.green(`${icon} ${testName}: ${message}`, details);
            // Send to remote debug
            if (window.VSRemoteDebug) {
                window.VSRemoteDebug.success('VSSiteVector', `${testName}: ${message}`, details);
            }
        } else if (status === 'fail') {
            TestResults.failed++;
            TestResults.issues.push({ test: testName, message, details });
            DiagnosticConsole.red(`${icon} ${testName}: ${message}`, details);
            // Send to remote debug
            if (window.VSRemoteDebug) {
                window.VSRemoteDebug.error('VSSiteVector', `${testName}: ${message}`, details);
            }
        } else {
            TestResults.warnings++;
            DiagnosticConsole.yellow(`${icon} ${testName}: ${message}`, details);
            // Send to remote debug
            if (window.VSRemoteDebug) {
                window.VSRemoteDebug.warn('VSSiteVector', `${testName}: ${message}`, details);
            }
        }

        return status === 'pass';
    }

    /**
     * Test AJAX endpoint
     */
    function testAjaxEndpoint(action, description, testData = {}) {
        return new Promise((resolve) => {
            const ajaxUrl = window.vssvDiag?.ajaxUrl || window.ajaxurl || '/wp-admin/admin-ajax.php';
            const nonce = window.vssvDiag?.nonce || '';

            $.ajax({
                url: ajaxUrl,
                type: 'POST',
                data: {
                    action: action,
                    nonce: nonce,
                    ...testData
                },
                timeout: 15000
            })
            .done(function(response) {
                if (response && response.success !== undefined) {
                    logTest(`AJAX: ${description}`, 'pass', 'Endpoint responds correctly', response.data);
                    resolve({ success: true, data: response.data });
                } else {
                    logTest(`AJAX: ${description}`, 'warn', 'Unexpected response format', response);
                    resolve({ success: false, data: response });
                }
            })
            .fail(function(xhr, status, error) {
                if (xhr.status === 403 || xhr.status === 401) {
                    logTest(`AJAX: ${description}`, 'pass', 'Properly secured (nonce validation)');
                    resolve({ success: true, secured: true });
                } else if (xhr.status === 400) {
                    logTest(`AJAX: ${description}`, 'pass', 'Endpoint validates input');
                    resolve({ success: true, validated: true });
                } else if (status === 'timeout') {
                    logTest(`AJAX: ${description}`, 'fail', 'Request timed out');
                    TestResults.fixes.push(`Check server response time for ${action}`);
                    resolve({ success: false, error: 'timeout' });
                } else {
                    logTest(`AJAX: ${description}`, 'fail', `${xhr.status}: ${error}`, xhr.responseText);
                    resolve({ success: false, error: error });
                }
            });
        });
    }

    /**
     * Test browser Performance API capabilities
     */
    function testPerformanceAPI() {
        DiagnosticConsole.blue('Testing Browser Performance APIs...');

        // Navigation Timing API
        if (window.performance && window.performance.timing) {
            const timing = window.performance.timing;
            const metrics = {
                dns: timing.domainLookupEnd - timing.domainLookupStart,
                tcp: timing.connectEnd - timing.connectStart,
                ttfb: timing.responseStart - timing.navigationStart,
                download: timing.responseEnd - timing.responseStart,
                domParse: timing.domInteractive - timing.responseEnd,
                domReady: timing.domContentLoadedEventEnd - timing.navigationStart,
                loadTime: timing.loadEventEnd - timing.navigationStart
            };
            TestResults.metrics.timing = metrics;
            logTest('Navigation Timing', 'pass', 'Available', metrics);
        } else {
            logTest('Navigation Timing', 'fail', 'Not available');
        }

        // Resource Timing API
        if (window.performance && window.performance.getEntriesByType) {
            const resources = window.performance.getEntriesByType('resource');
            logTest('Resource Timing', 'pass', `${resources.length} resources tracked`);
        } else {
            logTest('Resource Timing', 'warn', 'Not available');
        }

        // PerformanceObserver for Core Web Vitals
        if (window.PerformanceObserver) {
            logTest('PerformanceObserver', 'pass', 'Available for Core Web Vitals');

            const supportedTypes = [];
            ['largest-contentful-paint', 'first-input', 'layout-shift', 'longtask'].forEach(type => {
                try {
                    const observer = new PerformanceObserver(() => {});
                    observer.observe({ type, buffered: true });
                    observer.disconnect();
                    supportedTypes.push(type);
                } catch (e) {
                    // Not supported
                }
            });

            if (supportedTypes.length > 0) {
                logTest('Web Vitals Support', 'pass', `Supports: ${supportedTypes.join(', ')}`);
            } else {
                logTest('Web Vitals Support', 'warn', 'No Core Web Vitals entry types supported');
            }
        } else {
            logTest('PerformanceObserver', 'warn', 'Not available - limited metrics collection');
        }

        // Memory API (Chrome only)
        if (window.performance && window.performance.memory) {
            const memory = window.performance.memory;
            const memoryInfo = {
                usedJS: Math.round(memory.usedJSHeapSize / 1048576) + ' MB',
                totalJS: Math.round(memory.totalJSHeapSize / 1048576) + ' MB',
                limit: Math.round(memory.jsHeapSizeLimit / 1048576) + ' MB'
            };
            TestResults.metrics.memory = memoryInfo;
            logTest('Memory API', 'pass', 'Available', memoryInfo);
        } else {
            logTest('Memory API', 'warn', 'Not available (Chrome only)');
        }
    }

    /**
     * Test cache storage capabilities
     */
    function testCacheStorage() {
        DiagnosticConsole.blue('Testing Cache Storage...');

        // localStorage test
        try {
            const testKey = 'vssv_test_' + Date.now();
            localStorage.setItem(testKey, 'test');
            const retrieved = localStorage.getItem(testKey);
            localStorage.removeItem(testKey);

            if (retrieved === 'test') {
                logTest('localStorage', 'pass', 'Available and working');

                // Check for VSSV cached data
                const vssvKeys = Object.keys(localStorage).filter(k => k.startsWith('vssv_'));
                if (vssvKeys.length > 0) {
                    logTest('VSSV Local Data', 'pass', `${vssvKeys.length} cached items found`);
                }
            }
        } catch (e) {
            logTest('localStorage', 'fail', 'Not available', e.message);
            TestResults.fixes.push('Enable localStorage in browser settings');
        }

        // sessionStorage test
        try {
            const testKey = 'vssv_session_test_' + Date.now();
            sessionStorage.setItem(testKey, 'test');
            sessionStorage.removeItem(testKey);
            logTest('sessionStorage', 'pass', 'Available');
        } catch (e) {
            logTest('sessionStorage', 'warn', 'Not available', e.message);
        }

        // Service Worker / Cache API
        if ('caches' in window) {
            logTest('Cache API', 'pass', 'Service Worker cache available');

            caches.keys().then(names => {
                if (names.length > 0) {
                    logTest('Active Caches', 'pass', `${names.length} cache(s) found`, names);
                }
            }).catch(() => {
                // Silent fail
            });
        } else {
            logTest('Cache API', 'warn', 'Not available - no offline caching');
        }
    }

    /**
     * Test plugin configuration
     */
    function testPluginConfig() {
        DiagnosticConsole.blue('Testing Plugin Configuration...');

        // Check vssvConfig object
        if (typeof window.vssvConfig !== 'undefined') {
            logTest('vssvConfig Object', 'pass', 'Available', {
                version: window.vssvConfig.version,
                cacheEnabled: window.vssvConfig.cacheEnabled,
                optimizerActive: window.vssvConfig.optimizerActive
            });

            // Module status checks
            if (window.vssvConfig.modules) {
                const modules = window.vssvConfig.modules;
                Object.keys(modules).forEach(mod => {
                    const status = modules[mod] ? 'pass' : 'warn';
                    const msg = modules[mod] ? 'Active' : 'Inactive';
                    logTest(`Module: ${mod}`, status, msg);
                });
            }

            // Cache settings
            if (window.vssvConfig.cacheEnabled) {
                logTest('Page Caching', 'pass', 'Enabled');
            } else {
                logTest('Page Caching', 'warn', 'Disabled - consider enabling for better performance');
                TestResults.fixes.push('Enable page caching in VS Site Vector settings');
            }

        } else {
            logTest('vssvConfig Object', 'warn', 'Not found on frontend - admin config only');
        }
    }

    /**
     * Test server diagnostics via AJAX (admin only)
     */
    async function testServerDiagnostics() {
        DiagnosticConsole.blue('Testing Server Environment...');

        if (!window.vssvDiag?.ajaxUrl || !window.vssvDiag?.isAdmin) {
            logTest('Server Diagnostics', 'warn', 'Admin access required for full diagnostics');
            return;
        }

        const result = await testAjaxEndpoint('vssv_diagnostic_check', 'Server Diagnostics');

        if (result.success && result.data) {
            const data = result.data;

            // PHP Version
            if (data.php_version) {
                const phpOk = parseFloat(data.php_version) >= 7.4;
                logTest('PHP Version', phpOk ? 'pass' : 'fail', data.php_version,
                    phpOk ? null : 'PHP 7.4+ required');
            }

            // WordPress Version
            if (data.wp_version) {
                const wpOk = parseFloat(data.wp_version) >= 5.8;
                logTest('WordPress Version', wpOk ? 'pass' : 'warn', data.wp_version);
            }

            // Database Tables
            if (data.tables_exist) {
                logTest('Database Tables', 'pass', 'All tables exist');
            } else {
                logTest('Database Tables', 'fail', 'Missing tables', data.missing_tables);
                TestResults.fixes.push('Deactivate and reactivate VS Site Vector to create tables');
            }

            // Cache Directory
            if (data.cache_writable) {
                logTest('Cache Directory', 'pass', 'Writable');
            } else {
                logTest('Cache Directory', 'fail', 'Not writable');
                TestResults.fixes.push('Set write permissions on wp-content/cache/vs-site-vector/');
            }

            // PHP Extensions
            if (data.extensions) {
                const required = ['mbstring', 'dom', 'xml', 'json'];
                const recommended = ['imagick', 'redis', 'apcu', 'opcache'];

                required.forEach(ext => {
                    if (data.extensions.required && data.extensions.required[ext]) {
                        logTest(`PHP Extension: ${ext}`, 'pass', 'Installed');
                    } else {
                        logTest(`PHP Extension: ${ext}`, 'fail', 'Missing (required)');
                        TestResults.fixes.push(`Install PHP ${ext} extension`);
                    }
                });

                recommended.forEach(ext => {
                    if (data.extensions.recommended && data.extensions.recommended[ext]) {
                        logTest(`PHP Extension: ${ext}`, 'pass', 'Installed (recommended)');
                    } else {
                        logTest(`PHP Extension: ${ext}`, 'warn', 'Not installed (recommended)');
                    }
                });
            }

            // Active Modules
            if (data.active_modules) {
                logTest('Active Modules', 'pass', `${data.active_modules.length} modules active`,
                    data.active_modules);
            }

            // Performance Score
            if (data.performance_score !== undefined) {
                const scoreStatus = data.performance_score >= 80 ? 'pass' :
                    data.performance_score >= 50 ? 'warn' : 'fail';
                logTest('Performance Score', scoreStatus, `${data.performance_score}/100`);
                TestResults.metrics.performanceScore = data.performance_score;
            }

            // Cache Statistics
            if (data.cache_stats) {
                const stats = data.cache_stats;
                const hitRate = stats.hits && stats.total ?
                    Math.round((stats.hits / stats.total) * 100) : 0;

                logTest('Cache Hit Rate', hitRate >= 70 ? 'pass' : 'warn', `${hitRate}%`, {
                    hits: stats.hits,
                    misses: stats.misses,
                    total: stats.total
                });
                TestResults.metrics.cacheStats = stats;
            }

            // Recent Optimizations
            if (data.recent_optimizations !== undefined) {
                logTest('Recent Optimizations', 'pass',
                    `${data.recent_optimizations} in last 7 days`);
            }
        }
    }

    /**
     * Test cache functionality
     */
    async function testCacheFunctionality() {
        DiagnosticConsole.blue('Testing Cache System...');

        if (!window.vssvDiag?.ajaxUrl || !window.vssvDiag?.isAdmin) {
            logTest('Cache Tests', 'warn', 'Admin access required');
            return;
        }

        // Test cache status endpoint
        const cacheResult = await testAjaxEndpoint('vssv_get_cache_status', 'Cache Status');

        if (cacheResult.success && cacheResult.data) {
            const cache = cacheResult.data;

            // Page Cache
            if (cache.page_cache) {
                logTest('Page Cache', cache.page_cache.enabled ? 'pass' : 'warn',
                    cache.page_cache.enabled ? 'Enabled' : 'Disabled', {
                        entries: cache.page_cache.entries,
                        size: cache.page_cache.size
                    });
            }

            // Object Cache
            if (cache.object_cache) {
                logTest('Object Cache', cache.object_cache.enabled ? 'pass' : 'warn',
                    cache.object_cache.type || 'File-based', {
                        persistent: cache.object_cache.persistent
                    });
            }

            // Fragment Cache
            if (cache.fragment_cache) {
                logTest('Fragment Cache', cache.fragment_cache.enabled ? 'pass' : 'warn',
                    cache.fragment_cache.enabled ? 'Enabled' : 'Disabled');
            }
        }
    }

    /**
     * Test optimization features
     */
    async function testOptimizationFeatures() {
        DiagnosticConsole.blue('Testing Optimization Features...');

        // Check page optimizations applied
        const html = document.documentElement.outerHTML;

        // Minification check
        const hasExcessiveWhitespace = html.split('\n').filter(l => l.trim() === '').length > 50;
        logTest('HTML Minification', !hasExcessiveWhitespace ? 'pass' : 'warn',
            !hasExcessiveWhitespace ? 'HTML appears minified' : 'HTML may not be minified');

        // Check for lazy loading
        const lazyImages = document.querySelectorAll('img[loading="lazy"]');
        const allImages = document.querySelectorAll('img');
        if (allImages.length > 0) {
            const lazyPercent = Math.round((lazyImages.length / allImages.length) * 100);
            logTest('Lazy Loading', lazyPercent >= 50 ? 'pass' : 'warn',
                `${lazyPercent}% of images have lazy loading`, {
                    lazy: lazyImages.length,
                    total: allImages.length
                });
        }

        // Check for defer/async scripts
        const scripts = document.querySelectorAll('script[src]');
        const asyncDefer = document.querySelectorAll('script[async], script[defer]');
        if (scripts.length > 0) {
            const optimizedPercent = Math.round((asyncDefer.length / scripts.length) * 100);
            logTest('Script Optimization', optimizedPercent >= 30 ? 'pass' : 'warn',
                `${optimizedPercent}% scripts are async/defer`, {
                    optimized: asyncDefer.length,
                    total: scripts.length
                });
        }

        // Check for preconnect/dns-prefetch
        const preconnects = document.querySelectorAll('link[rel="preconnect"], link[rel="dns-prefetch"]');
        logTest('Resource Hints', preconnects.length > 0 ? 'pass' : 'warn',
            preconnects.length > 0 ? `${preconnects.length} preconnect/dns-prefetch hints` : 'No resource hints found');

        // Check for critical CSS
        const inlineStyles = document.querySelectorAll('style[data-critical], style#vssv-critical');
        logTest('Critical CSS', inlineStyles.length > 0 ? 'pass' : 'warn',
            inlineStyles.length > 0 ? 'Critical CSS inlined' : 'No critical CSS detected');
    }

    /**
     * Test static site generation (admin only)
     */
    async function testStaticGeneration() {
        DiagnosticConsole.blue('Testing Static Site Generation...');

        if (!window.vssvDiag?.ajaxUrl || !window.vssvDiag?.isAdmin) {
            logTest('Static Generation', 'warn', 'Admin access required');
            return;
        }

        const result = await testAjaxEndpoint('vssv_get_static_status', 'Static Generation Status');

        if (result.success && result.data) {
            const static_ = result.data;

            logTest('Static Generation', static_.enabled ? 'pass' : 'warn',
                static_.enabled ? 'Enabled' : 'Disabled');

            if (static_.enabled) {
                if (static_.pages_generated !== undefined) {
                    logTest('Generated Pages', 'pass', `${static_.pages_generated} pages`);
                }

                if (static_.last_build) {
                    logTest('Last Build', 'pass', static_.last_build);
                }

                if (static_.auto_rebuild) {
                    logTest('Auto Rebuild', 'pass', 'Enabled');
                } else {
                    logTest('Auto Rebuild', 'warn', 'Disabled');
                }
            }
        }
    }

    /**
     * Test CDN deployment (admin only)
     */
    async function testCDNDeployment() {
        DiagnosticConsole.blue('Testing CDN Deployment...');

        if (!window.vssvDiag?.ajaxUrl || !window.vssvDiag?.isAdmin) {
            logTest('CDN Deployment', 'warn', 'Admin access required');
            return;
        }

        const result = await testAjaxEndpoint('vssv_get_deploy_status', 'CDN Status');

        if (result.success && result.data) {
            const cdn = result.data;

            if (cdn.configured) {
                logTest('CDN Configuration', 'pass', 'Configured', {
                    provider: cdn.provider
                });

                if (cdn.connected) {
                    logTest('CDN Connection', 'pass', 'Connected');
                } else {
                    logTest('CDN Connection', 'fail', 'Not connected');
                    TestResults.fixes.push('Check CDN credentials in settings');
                }

                if (cdn.last_deploy) {
                    logTest('Last Deployment', 'pass', cdn.last_deploy);
                }
            } else {
                logTest('CDN Configuration', 'warn', 'Not configured');
            }
        }
    }

    /**
     * Run all AJAX endpoint tests
     */
    async function testAllAjaxEndpoints() {
        DiagnosticConsole.blue('Testing AJAX Endpoints...');

        const endpoints = [
            { action: 'vssv_run_optimization', desc: 'Run Optimization' },
            { action: 'vssv_clear_cache', desc: 'Clear Cache' },
            { action: 'vssv_get_performance_data', desc: 'Get Performance Data' }
        ];

        for (const endpoint of endpoints) {
            await testAjaxEndpoint(endpoint.action, endpoint.desc);
        }
    }

    /**
     * Quick action: Clear all caches
     */
    async function clearAllCaches() {
        DiagnosticConsole.purple('Clearing all caches...');

        if (!window.vssvDiag?.ajaxUrl) {
            DiagnosticConsole.red('Cannot clear caches - admin access required');
            return false;
        }

        const result = await testAjaxEndpoint('vssv_clear_cache', 'Clear Cache');

        if (result.success) {
            DiagnosticConsole.green('All caches cleared! Refresh the page to see effects.');
            return true;
        } else {
            DiagnosticConsole.red('Failed to clear caches');
            return false;
        }
    }

    /**
     * Quick action: Run optimization
     */
    async function runOptimization() {
        DiagnosticConsole.purple('Running optimization...');

        if (!window.vssvDiag?.ajaxUrl) {
            DiagnosticConsole.red('Cannot run optimization - admin access required');
            return false;
        }

        const result = await testAjaxEndpoint('vssv_run_optimization', 'Run Optimization');

        if (result.success) {
            DiagnosticConsole.green('Optimization completed!', result.data);
            return true;
        } else {
            DiagnosticConsole.red('Optimization failed');
            return false;
        }
    }

    /**
     * Get page load metrics
     */
    function getPageMetrics() {
        const metrics = {};

        if (window.performance && window.performance.timing) {
            const t = window.performance.timing;
            metrics.dns = t.domainLookupEnd - t.domainLookupStart;
            metrics.tcp = t.connectEnd - t.connectStart;
            metrics.ttfb = t.responseStart - t.navigationStart;
            metrics.download = t.responseEnd - t.responseStart;
            metrics.domReady = t.domContentLoadedEventEnd - t.navigationStart;
            metrics.loadTime = t.loadEventEnd - t.navigationStart;
        }

        metrics.resources = window.performance?.getEntriesByType?.('resource')?.length || 0;

        return metrics;
    }

    /**
     * Print diagnostic summary
     */
    function printSummary() {
        DiagnosticConsole.separator();
        DiagnosticConsole.purple('VS Site Vector Diagnostic Summary');
        DiagnosticConsole.separator();

        const total = TestResults.passed + TestResults.failed + TestResults.warnings;

        // Removed console.log statements

        // Send complete results to remote debug
        if (window.VSRemoteDebug && window.VSRemoteDebug.isConnected()) {
            window.VSRemoteDebug.sendDiagnostics('VSSiteVector', {
                passed: TestResults.passed,
                failed: TestResults.failed,
                warnings: TestResults.warnings,
                issues: TestResults.issues,
                fixes: TestResults.fixes,
                metrics: TestResults.metrics
            });
        }
    }

    /**
     * Run all diagnostic tests
     */
    async function runDiagnostics() {
        // Reset results
        TestResults.passed = 0;
        TestResults.failed = 0;
        TestResults.warnings = 0;
        TestResults.issues = [];
        TestResults.fixes = [];
        TestResults.metrics = {};

        // Removed console output
        DiagnosticConsole.purple('VS Site Vector Diagnostic Suite v1.0');
        DiagnosticConsole.separator();

        // Browser-side tests (always run)
        testPerformanceAPI();
        DiagnosticConsole.separator();

        testCacheStorage();
        DiagnosticConsole.separator();

        testPluginConfig();
        DiagnosticConsole.separator();

        testOptimizationFeatures();
        DiagnosticConsole.separator();

        // Server-side tests (admin only)
        await testServerDiagnostics();
        DiagnosticConsole.separator();

        await testCacheFunctionality();
        DiagnosticConsole.separator();

        await testStaticGeneration();
        DiagnosticConsole.separator();

        await testCDNDeployment();
        DiagnosticConsole.separator();

        // AJAX endpoint tests
        await testAllAjaxEndpoints();

        // Print summary
        printSummary();
    }

    // Expose diagnostic functions globally
    window.VSSVDiag = {
        runTests: runDiagnostics,
        clearCache: clearAllCaches,
        optimize: runOptimization,
        getMetrics: getPageMetrics,
        testAjax: testAllAjaxEndpoints,
        results: TestResults
    };

    // Run diagnostics when ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', () => {
            setTimeout(runDiagnostics, 500);
        });
    } else {
        setTimeout(runDiagnostics, 500);
    }

})();
