/**
 * VS Site Vector - Performance Tracker
 *
 * Captures real Web Vitals metrics and sends to server
 * No cookies - anonymous visitor hash for deduplication only
 *
 * @package VS_Site_Vector
 * @since 2.0.0
 */

(function() {
    'use strict';

    // Configuration - injected by WordPress
    var config = window.vssvTrackerConfig || {};
    var ajaxUrl = config.ajaxUrl || '/wp-admin/admin-ajax.php';
    var token = config.token || generateToken();
    var DEBUG = config.debug || false;

    function log() {
        // Removed console.log
    }

    /**
     * Generate a random token for this page load
     */
    function generateToken() {
        return Math.random().toString(36).substring(2, 15);
    }

    /**
     * Detect device type based on screen width
     */
    function getDeviceType() {
        var width = window.innerWidth || document.documentElement.clientWidth;
        if (width < 768) return 'mobile';
        if (width < 1024) return 'tablet';
        return 'desktop';
    }

    /**
     * Get connection information
     * Returns actual speed in Mbps and a meaningful connection category
     */
    function getConnectionInfo() {
        var conn = navigator.connection || navigator.mozConnection || navigator.webkitConnection;
        if (!conn) return { type: null, speed: null, rtt: null };

        var speed = conn.downlink || null;  // Mbps
        var rtt = conn.rtt || null;         // ms

        // Determine connection type based on actual speed
        var type = null;
        if (speed !== null) {
            if (speed >= 100) {
                type = 'fiber';           // 100+ Mbps - Fiber/Cable
            } else if (speed >= 25) {
                type = 'broadband';       // 25-100 Mbps - Good broadband
            } else if (speed >= 10) {
                type = '5g';              // 10-25 Mbps - 5G or fast 4G
            } else if (speed >= 4) {
                type = '4g';              // 4-10 Mbps - Typical 4G
            } else if (speed >= 1) {
                type = '3g';              // 1-4 Mbps - 3G
            } else {
                type = 'slow';            // < 1 Mbps - 2G or poor connection
            }
        } else if (conn.effectiveType) {
            // Fallback to effectiveType if speed not available
            type = conn.effectiveType;
        }

        return {
            type: type,
            speed: speed,
            rtt: rtt
        };
    }

    /**
     * Create metrics object for current page load
     */
    function createMetricsEntry() {
        var connInfo = getConnectionInfo();
        return {
            url: window.location.pathname + window.location.search,
            metrics: {
                lcp: null,
                fid: null,
                cls: null,
                ttfb: null,
                fcp: null,
                dom_load: null,
                page_load: null
            },
            connection_type: connInfo.type,
            connection_speed: connInfo.speed,  // Mbps
            connection_rtt: connInfo.rtt,      // ms
            device: getDeviceType()
        };
    }

    /**
     * Collect Navigation Timing metrics
     */
    function collectNavigationTiming(entry) {
        if (!performance || !performance.getEntriesByType) return;

        var navEntries = performance.getEntriesByType('navigation');
        if (!navEntries.length) return;

        var nav = navEntries[0];

        entry.metrics.ttfb = Math.round(nav.responseStart - nav.requestStart);
        entry.metrics.dom_load = Math.round(nav.domContentLoadedEventEnd - nav.startTime);
        entry.metrics.page_load = Math.round(nav.loadEventEnd - nav.startTime);
    }

    /**
     * Observe Largest Contentful Paint (LCP)
     */
    function observeLCP(entry, onUpdate) {
        if (!('PerformanceObserver' in window)) return;

        try {
            var observer = new PerformanceObserver(function(list) {
                var entries = list.getEntries();
                var lastEntry = entries[entries.length - 1];
                entry.metrics.lcp = Math.round(lastEntry.startTime);
                if (onUpdate) onUpdate();
            });

            observer.observe({ type: 'largest-contentful-paint', buffered: true });

            // Disconnect on page hide
            document.addEventListener('visibilitychange', function() {
                if (document.visibilityState === 'hidden') {
                    observer.disconnect();
                }
            }, { once: true });
        } catch (e) {
            // LCP not supported
        }
    }

    /**
     * Observe First Input Delay (FID)
     */
    function observeFID(entry, onUpdate) {
        if (!('PerformanceObserver' in window)) return;

        try {
            var observer = new PerformanceObserver(function(list) {
                var entries = list.getEntries();
                entries.forEach(function(fidEntry) {
                    entry.metrics.fid = Math.round(fidEntry.processingStart - fidEntry.startTime);
                    if (onUpdate) onUpdate();
                });
            });

            observer.observe({ type: 'first-input', buffered: true });
        } catch (e) {
            // FID not supported
        }
    }

    /**
     * Observe Cumulative Layout Shift (CLS)
     */
    function observeCLS(entry, onUpdate) {
        if (!('PerformanceObserver' in window)) return;

        var clsValue = 0;
        var sessionValue = 0;
        var sessionEntries = [];

        try {
            var observer = new PerformanceObserver(function(list) {
                list.getEntries().forEach(function(layoutShift) {
                    if (!layoutShift.hadRecentInput) {
                        if (sessionEntries.length > 0) {
                            var firstSessionEntry = sessionEntries[0];
                            var lastSessionEntry = sessionEntries[sessionEntries.length - 1];

                            if (layoutShift.startTime - lastSessionEntry.startTime > 1000 ||
                                layoutShift.startTime - firstSessionEntry.startTime > 5000) {
                                clsValue = Math.max(clsValue, sessionValue);
                                sessionValue = 0;
                                sessionEntries = [];
                            }
                        }

                        sessionValue += layoutShift.value;
                        sessionEntries.push(layoutShift);

                        clsValue = Math.max(clsValue, sessionValue);
                        entry.metrics.cls = Math.round(clsValue * 1000) / 1000;
                        if (onUpdate) onUpdate();
                    }
                });
            });

            observer.observe({ type: 'layout-shift', buffered: true });
        } catch (e) {
            // CLS not supported
        }
    }

    /**
     * Observe First Contentful Paint (FCP)
     */
    function observeFCP(entry, onUpdate) {
        if (!('PerformanceObserver' in window)) return;

        try {
            var observer = new PerformanceObserver(function(list) {
                var entries = list.getEntries();
                entries.forEach(function(fcpEntry) {
                    if (fcpEntry.name === 'first-contentful-paint') {
                        entry.metrics.fcp = Math.round(fcpEntry.startTime);
                        observer.disconnect();
                        if (onUpdate) onUpdate();
                    }
                });
            });

            observer.observe({ type: 'paint', buffered: true });
        } catch (e) {
            // FCP not supported
        }
    }

    /**
     * Send metrics to server
     */
    function sendToServer(entry) {
        log('Sending metrics:', entry);

        // Build form data
        var formData = new FormData();
        formData.append('action', 'vssv_collect_vitals');
        formData.append('token', token);
        formData.append('url', entry.url);

        // Add metrics
        if (entry.metrics.lcp !== null) formData.append('lcp', entry.metrics.lcp);
        if (entry.metrics.fid !== null) formData.append('fid', entry.metrics.fid);
        if (entry.metrics.cls !== null) formData.append('cls', entry.metrics.cls);
        if (entry.metrics.ttfb !== null) formData.append('ttfb', entry.metrics.ttfb);
        if (entry.metrics.fcp !== null) formData.append('fcp', entry.metrics.fcp);
        if (entry.metrics.dom_load !== null) formData.append('dom_load', entry.metrics.dom_load);
        if (entry.metrics.page_load !== null) formData.append('page_load', entry.metrics.page_load);

        // Add context
        if (entry.connection_type) formData.append('connection_type', entry.connection_type);
        if (entry.connection_speed !== null) formData.append('connection_speed', entry.connection_speed);
        if (entry.connection_rtt !== null) formData.append('connection_rtt', entry.connection_rtt);
        if (entry.device) formData.append('device', entry.device);

        // Use sendBeacon for reliability (doesn't block page unload)
        if (navigator.sendBeacon) {
            // Convert FormData to URLSearchParams for sendBeacon
            var params = new URLSearchParams();
            formData.forEach(function(value, key) {
                params.append(key, value);
            });

            var sent = navigator.sendBeacon(ajaxUrl, params);
            log('sendBeacon result:', sent, 'URL:', ajaxUrl);
        } else {
            // Fallback to XHR
            log('Using XHR fallback');
            var xhr = new XMLHttpRequest();
            xhr.open('POST', ajaxUrl, true);
            xhr.onload = function() {
                log('XHR response:', xhr.status, xhr.responseText);
            };
            xhr.onerror = function() {
                log('XHR error');
            };
            xhr.send(formData);
        }
    }

    /**
     * Main tracking initialization
     */
    function initTracker() {
        log('initTracker called, config:', config);

        // Don't track in admin
        if (document.body && document.body.classList.contains('wp-admin')) {
            log('Skipping - admin page detected');
            return;
        }

        var currentEntry = createMetricsEntry();
        var sent = false;

        log('Created entry for:', currentEntry.url);

        // Send metrics to server (only once)
        function sendMetrics() {
            if (sent) {
                log('Already sent, skipping');
                return;
            }
            sent = true;

            collectNavigationTiming(currentEntry);
            log('Collected navigation timing, sending...');
            sendToServer(currentEntry);
        }

        // Set up observers (they update entry.metrics as values become available)
        observeFCP(currentEntry);
        observeLCP(currentEntry);
        observeFID(currentEntry);
        observeCLS(currentEntry);

        // Send after page load + delay for LCP to finalize
        if (document.readyState === 'complete') {
            log('Page already complete, sending in 100ms');
            setTimeout(sendMetrics, 100);
        } else {
            log('Waiting for page load...');
            window.addEventListener('load', function() {
                // Wait for LCP to finalize (up to 3 seconds)
                log('Page loaded, sending in 3s');
                setTimeout(sendMetrics, 3000);
            });
        }

        // Also send on page hide (catches tab close, navigation)
        document.addEventListener('visibilitychange', function() {
            if (document.visibilityState === 'hidden') {
                log('Page hidden, sending metrics');
                sendMetrics();
            }
        });
    }

    // Initialize on DOM ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', initTracker);
    } else {
        initTracker();
    }

})();
