/**
 * VS Site Vector - Custom UI JavaScript
 * Core application functionality for the custom interface
 */

(function() {
    'use strict';

    // Application State
    const VSSVApp = {
        config: window.vssvConfig || {},
        currentSection: 'dashboard',

        // Initialize the application
        init() {
            this.bindEvents();
            this.initializeComponents();
            this.loadDashboard();
        },

        // Bind all event listeners
        bindEvents() {
            // Navigation
            this.bindNavigation();

            // Mobile menu
            this.bindMobileMenu();

            // Window resize
            window.addEventListener('resize', this.handleResize.bind(this));
        },

        // Bind navigation events
        bindNavigation() {
            document.querySelectorAll('.vssv-nav-item').forEach(item => {
                item.addEventListener('click', (e) => {
                    e.preventDefault();
                    this.navigateToSection(item.dataset.section);
                });
            });
        },

        // Navigate to a section
        navigateToSection(section) {
            // Update active nav
            document.querySelectorAll('.vssv-nav-item').forEach(item => {
                item.classList.toggle('active', item.dataset.section === section);
            });

            // Show corresponding section
            document.querySelectorAll('.vssv-section').forEach(s => {
                s.classList.remove('active');
            });

            const targetSection = document.getElementById(section + '-section');
            if (targetSection) {
                targetSection.classList.add('active');
                this.currentSection = section;

                // Load section data if needed
                this.loadSectionData(section);
            }
        },

        // Load data for a specific section
        loadSectionData(section) {
            switch(section) {
                case 'dashboard':
                    this.loadDashboard();
                    break;
                case 'static':
                    this.loadStaticGeneration();
                    break;
                case 'deployment':
                    this.loadDeployment();
                    break;
                case 'cache':
                    this.loadCacheManager();
                    break;
                case 'optimization':
                    this.loadOptimization();
                    break;
                case 'monitor':
                    this.loadMonitor();
                    break;
                case 'settings':
                    this.loadSettings();
                    break;
            }
        },

        // Initialize UI components
        initializeComponents() {
            // Initialize performance score animation
            this.initPerformanceScore();

            // Initialize tooltips if needed
            this.initTooltips();

            // Initialize modals if needed
            this.initModals();
        },

        // Initialize performance score circle
        initPerformanceScore() {
            const scoreElement = document.getElementById('scoreNumber');
            const circle = document.getElementById('scoreCircle');

            if (scoreElement && circle) {
                const score = parseInt(scoreElement.textContent) || 0;
                const offset = 565 - (565 * score) / 100;

                setTimeout(() => {
                    circle.style.strokeDashoffset = offset;

                    // Animate number
                    this.animateNumber(scoreElement, 0, score, 1000);
                }, 100);
            }
        },

        // Animate number from start to end
        animateNumber(element, start, end, duration) {
            const range = end - start;
            const increment = range / (duration / 16);
            let current = start;

            const timer = setInterval(() => {
                current += increment;
                if (current >= end) {
                    current = end;
                    clearInterval(timer);
                }
                element.textContent = Math.round(current);
            }, 16);
        },

        // Mobile menu functionality
        bindMobileMenu() {
            const toggle = document.getElementById('vssvMobileToggle');
            const sidebar = document.querySelector('.vssv-sidebar');
            const overlay = document.getElementById('vssvOverlay');

            if (toggle && sidebar && overlay) {
                toggle.addEventListener('click', () => {
                    sidebar.classList.toggle('open');
                    overlay.classList.toggle('active');
                });

                overlay.addEventListener('click', () => {
                    sidebar.classList.remove('open');
                    overlay.classList.remove('active');
                });
            }
        },

        // Handle window resize
        handleResize() {
            // Close mobile menu on desktop resize
            if (window.innerWidth > 768) {
                document.querySelector('.vssv-sidebar')?.classList.remove('open');
                document.getElementById('vssvOverlay')?.classList.remove('active');
            }
        },

        // Initialize tooltips
        initTooltips() {
            // Placeholder for tooltip initialization
        },

        // Initialize modals
        initModals() {
            // Placeholder for modal initialization
        },

        // Load dashboard data
        loadDashboard() {
            // Dashboard is already loaded, but we can refresh data here if needed
            this.refreshMetrics();
        },

        // Refresh dashboard metrics
        async refreshMetrics() {
            try {
                const data = await this.api('get_metrics');
                if (data.success) {
                    this.updateMetrics(data.data);
                }
            } catch (error) {
                console.error('Failed to refresh metrics:', error);
            }
        },

        // Update metrics display
        updateMetrics(metrics) {
            // Update load time
            const loadTimeEl = document.querySelector('[data-metric="load-time"]');
            if (loadTimeEl && metrics.load_time) {
                loadTimeEl.textContent = metrics.load_time;
            }

            // Update cache hit rate
            const cacheHitEl = document.querySelector('[data-metric="cache-hit"]');
            if (cacheHitEl && metrics.cache_hit_rate) {
                cacheHitEl.textContent = metrics.cache_hit_rate + '%';
            }

            // Update other metrics as needed
        },

        // Load static generation section
        loadStaticGeneration() {
            // Load static generation data
        },

        // Load deployment section
        loadDeployment() {
            // Load deployment data
        },

        // Load cache manager section
        loadCacheManager() {
            // Load cache data
        },

        // Load optimization section
        loadOptimization() {
            // Load optimization data
        },

        // Load monitor section
        loadMonitor() {
            // Load monitoring data
        },

        // Load settings section
        loadSettings() {
            // Load settings data
        },

        // API wrapper function
        async api(endpoint, data = {}) {
            const formData = new FormData();
            formData.append('action', 'vssv_api');
            formData.append('nonce', this.config.nonce);
            formData.append('endpoint', endpoint);

            Object.keys(data).forEach(key => {
                formData.append(key, data[key]);
            });

            const response = await fetch(this.config.ajaxUrl, {
                method: 'POST',
                body: formData
            });

            return response.json();
        }
    };

    // Global action functions
    window.vssvOptimize = async function() {
        try {
            showLoading('Optimizing...');
            const result = await VSSVApp.api('optimize');
            hideLoading();

            if (result.success) {
                showNotification('Optimization completed successfully!', 'success');
                VSSVApp.refreshMetrics();
            } else {
                showNotification('Optimization failed. Please try again.', 'error');
            }
        } catch (error) {
            hideLoading();
            showNotification('An error occurred during optimization.', 'error');
            console.error('Optimization error:', error);
        }
    };

    window.vssvGenerateStatic = async function() {
        const progress = document.getElementById('staticProgress');
        const progressBar = document.getElementById('staticProgressBar');
        const progressText = document.getElementById('staticProgressText');

        if (!progress || !progressBar || !progressText) return;

        progress.classList.add('active');

        try {
            // Start generation
            const result = await VSSVApp.api('generate_static');

            if (result.success) {
                // Real progress tracking
                let currentProgress = 0;
                const updateProgress = () => {
                    if (result.data && result.data.progress) {
                        currentProgress = result.data.progress;
                    } else {
                        // Simulate progress
                        currentProgress += 10;
                    }

                    progressBar.style.width = currentProgress + '%';
                    progressText.textContent = currentProgress + '%';

                    if (currentProgress < 100) {
                        setTimeout(updateProgress, 500);
                    } else {
                        setTimeout(() => {
                            progress.classList.remove('active');
                            showNotification('Static generation completed!', 'success');
                        }, 500);
                    }
                };

                updateProgress();
            }
        } catch (error) {
            progress.classList.remove('active');
            showNotification('Static generation failed.', 'error');
            console.error('Static generation error:', error);
        }
    };

    window.vssvDeploy = async function() {
        try {
            showLoading('Deploying...');
            const result = await VSSVApp.api('deploy');
            hideLoading();

            if (result.success) {
                showNotification('Deployment triggered successfully!', 'success');
            } else {
                showNotification('Deployment failed. Please check your settings.', 'error');
            }
        } catch (error) {
            hideLoading();
            showNotification('An error occurred during deployment.', 'error');
            console.error('Deployment error:', error);
        }
    };

    window.vssvRefresh = function() {
        location.reload();
    };

    // Utility functions
    function showLoading(message = 'Loading...') {
        // Create loading overlay
        const overlay = document.createElement('div');
        overlay.className = 'vssv-loading-overlay';
        overlay.innerHTML = `
            <div class="vssv-loading-content">
                <div class="vssv-spinner"></div>
                <div class="vssv-loading-message">${message}</div>
            </div>
        `;
        document.body.appendChild(overlay);
    }

    function hideLoading() {
        const overlay = document.querySelector('.vssv-loading-overlay');
        if (overlay) {
            overlay.remove();
        }
    }

    function showNotification(message, type = 'info') {
        // Create notification element
        const notification = document.createElement('div');
        notification.className = `vssv-notification vssv-notification-${type}`;
        notification.innerHTML = `
            <div class="vssv-notification-content">
                <div class="vssv-notification-message">${message}</div>
                <button class="vssv-notification-close">&times;</button>
            </div>
        `;

        // Add to page
        document.body.appendChild(notification);

        // Auto remove after 5 seconds
        setTimeout(() => {
            notification.classList.add('fade-out');
            setTimeout(() => notification.remove(), 300);
        }, 5000);

        // Close button
        notification.querySelector('.vssv-notification-close').addEventListener('click', () => {
            notification.remove();
        });
    }

    // Initialize when DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', () => VSSVApp.init());
    } else {
        VSSVApp.init();
    }

    // Export for use in other scripts if needed
    window.VSSVApp = VSSVApp;

})();