<?php
/**
 * VS Site Vector API Module
 *
 * Production-safe minimal stub
 *
 * @package VS_Site_Vector
 * @since 1.2.1
 */

if (!defined('ABSPATH')) {
    exit;
}

class VSSV_API {

    /**
     * Constructor
     */
    public function __construct() {
        // Register only essential, lightweight REST endpoints
        add_action('rest_api_init', [$this, 'register_minimal_api_routes']);

        // Register AJAX handler for frontend API calls
        add_action('wp_ajax_vssv_api', [$this, 'handle_ajax_api']);
    }

    /**
     * Register minimal REST API routes - production safe
     * Only simple endpoints with no heavy operations
     */
    public function register_minimal_api_routes() {
        // GET /wp-json/vs/v1/site-vector/health - Simple health check
        register_rest_route('vs/v1/site-vector', '/health', [
            'methods' => 'GET',
            'callback' => [$this, 'api_health_check'],
            'permission_callback' => [$this, 'check_api_permission']
        ]);

        // GET /wp-json/vs/v1/site-vector/metrics - Basic metrics (cached)
        register_rest_route('vs/v1/site-vector', '/metrics', [
            'methods' => 'GET',
            'callback' => [$this, 'api_get_metrics'],
            'permission_callback' => [$this, 'check_api_permission']
        ]);

        // GET /wp-json/vs/v1/site-vector/cache/stats - Cache statistics
        register_rest_route('vs/v1/site-vector', '/cache/stats', [
            'methods' => 'GET',
            'callback' => [$this, 'api_get_cache_stats'],
            'permission_callback' => [$this, 'check_api_permission']
        ]);

        // POST /wp-json/vs/v1/site-vector/cache/purge - Simple cache purge
        register_rest_route('vs/v1/site-vector', '/cache/purge', [
            'methods' => 'POST',
            'callback' => [$this, 'api_purge_cache'],
            'permission_callback' => [$this, 'check_api_permission']
        ]);

        // GET /wp-json/vs/v1/site-vector/optimization/status - Optimization status
        register_rest_route('vs/v1/site-vector', '/optimization/status', [
            'methods' => 'GET',
            'callback' => [$this, 'api_get_optimization_status'],
            'permission_callback' => [$this, 'check_api_permission']
        ]);

        // GET /wp-json/vs/v1/site-vector/stats - Dashboard summary stats
        register_rest_route('vs/v1/site-vector', '/stats', [
            'methods' => 'GET',
            'callback' => [$this, 'api_get_stats'],
            'permission_callback' => [$this, 'check_api_permission']
        ]);

        // GET /wp-json/vs/v1/site-vector/recommendations - Simple recommendations
        register_rest_route('vs/v1/site-vector', '/recommendations', [
            'methods' => 'GET',
            'callback' => [$this, 'api_get_recommendations'],
            'permission_callback' => [$this, 'check_api_permission']
        ]);
    }

    /**
     * API Health Check - Ultra-lightweight
     */
    public function api_health_check($request) {
        return new WP_REST_Response([
            'status' => 'ok',
            'timestamp' => current_time('c'),
        ], 200);
    }

    /**
     * GET /vs/v1/site-vector/metrics - Basic metrics from options (no DB queries)
     */
    public function api_get_metrics($request) {
        try {
            $metrics = [
                'score' => (int) get_option('vssv_performance_score', 0),
                'load_time' => (float) get_option('vssv_avg_load_time', 0),
                'cache_hit_rate' => (float) get_option('vssv_cache_hit_rate', 0),
                'db_queries' => (int) get_option('vssv_avg_queries', 0),
            ];

            return new WP_REST_Response([
                'success' => true,
                'data' => $metrics
            ], 200);
        } catch (Exception $e) {
            return new WP_REST_Response([
                'success' => false,
                'error' => 'Failed to retrieve metrics'
            ], 500);
        }
    }

    /**
     * GET /vs/v1/site-vector/cache/stats - Cache statistics from options
     */
    public function api_get_cache_stats($request) {
        try {
            $stats = get_option('vssv_cache_stats', [
                'hits' => 0,
                'misses' => 0,
                'size' => 0,
            ]);

            $total = ($stats['hits'] ?? 0) + ($stats['misses'] ?? 0);
            $hit_rate = $total > 0 ? round(($stats['hits'] / $total) * 100, 2) : 0;

            return new WP_REST_Response([
                'success' => true,
                'data' => array_merge($stats, ['hit_rate' => $hit_rate])
            ], 200);
        } catch (Exception $e) {
            return new WP_REST_Response([
                'success' => false,
                'error' => 'Failed to retrieve cache stats'
            ], 500);
        }
    }

    /**
     * POST /vs/v1/site-vector/cache/purge - Clear cache safely
     */
    public function api_purge_cache($request) {
        try {
            // Clear WordPress object cache
            wp_cache_flush();

            // Reset cache stats
            update_option('vssv_cache_stats', ['hits' => 0, 'misses' => 0, 'size' => 0]);

            return new WP_REST_Response([
                'success' => true,
                'message' => 'Cache purged successfully',
                'timestamp' => current_time('c')
            ], 200);
        } catch (Exception $e) {
            return new WP_REST_Response([
                'success' => false,
                'error' => 'Failed to purge cache'
            ], 500);
        }
    }

    /**
     * GET /vs/v1/site-vector/optimization/status - Optimization status
     */
    public function api_get_optimization_status($request) {
        try {
            $last_optimization = get_option('vssv_last_optimization_time', 0);
            $results = get_option('vssv_last_optimization_results', []);

            $status = 'never_run';
            if ($last_optimization) {
                $age = time() - $last_optimization;
                // Status: recent (< 24h), stale (> 24h), or never_run
                $status = $age < 86400 ? 'recent' : 'stale';
            }

            return new WP_REST_Response([
                'success' => true,
                'data' => [
                    'status' => $status,
                    'last_run' => $last_optimization ? date('c', $last_optimization) : null,
                    'last_run_unix' => $last_optimization,
                    'results' => is_array($results) ? $results : [],
                ]
            ], 200);
        } catch (Exception $e) {
            return new WP_REST_Response([
                'success' => false,
                'error' => 'Failed to retrieve optimization status'
            ], 500);
        }
    }

    /**
     * GET /vs/v1/site-vector/stats - Dashboard summary (all key metrics in one call)
     */
    public function api_get_stats($request) {
        try {
            // Get cache stats
            $cache_stats = get_option('vssv_cache_stats', ['hits' => 0, 'misses' => 0]);
            $total_cache_requests = ($cache_stats['hits'] ?? 0) + ($cache_stats['misses'] ?? 0);
            $cache_hit_rate = $total_cache_requests > 0
                ? round(($cache_stats['hits'] / $total_cache_requests) * 100, 2)
                : 0;

            // Get optimization status
            $last_optimization = get_option('vssv_last_optimization_time', 0);

            // Build comprehensive stats object
            $stats = [
                'performance_score' => (int) get_option('vssv_performance_score', 0),
                'avg_load_time' => (float) get_option('vssv_avg_load_time', 0),
                'cache_hit_rate' => (float) $cache_hit_rate,
                'total_cache_requests' => (int) $total_cache_requests,
                'cache_hits' => (int) ($cache_stats['hits'] ?? 0),
                'cache_misses' => (int) ($cache_stats['misses'] ?? 0),
                'last_optimization' => $last_optimization ? date('c', $last_optimization) : null,
                'last_optimization_unix' => (int) $last_optimization,
                'avg_queries' => (int) get_option('vssv_avg_queries', 0),
                'memory_usage' => [
                    'peak' => memory_get_peak_usage(true),
                    'current' => memory_get_usage(true),
                    'limit' => ini_get('memory_limit'),
                ],
            ];

            return new WP_REST_Response([
                'success' => true,
                'data' => $stats
            ], 200);
        } catch (Exception $e) {
            return new WP_REST_Response([
                'success' => false,
                'error' => 'Failed to retrieve statistics'
            ], 500);
        }
    }

    /**
     * GET /vs/v1/site-vector/recommendations - Simple recommendations based on current metrics
     * Note: This version is simplified and doesn't call module methods to avoid timeouts
     */
    public function api_get_recommendations($request) {
        try {
            $recommendations = [];

            // Get current metrics
            $score = (int) get_option('vssv_performance_score', 0);
            $cache_stats = get_option('vssv_cache_stats', ['hits' => 0, 'misses' => 0]);
            $total_cache = ($cache_stats['hits'] ?? 0) + ($cache_stats['misses'] ?? 0);
            $hit_rate = $total_cache > 0 ? ($cache_stats['hits'] / $total_cache) * 100 : 0;
            $load_time = (float) get_option('vssv_avg_load_time', 0);
            $last_optimization = get_option('vssv_last_optimization_time', 0);

            // Recommendation 1: Performance Score
            if ($score < 50) {
                $recommendations[] = [
                    'type' => 'critical',
                    'title' => 'Critical Performance Issue',
                    'description' => 'Performance score is below 50. Immediate optimization needed.',
                    'priority' => 'high',
                    'action' => 'Run full optimization'
                ];
            } elseif ($score < 75) {
                $recommendations[] = [
                    'type' => 'warning',
                    'title' => 'Performance Below Optimal',
                    'description' => sprintf('Performance score is %d. Recommended target is 80+.', $score),
                    'priority' => 'medium',
                    'action' => 'Review optimization settings'
                ];
            }

            // Recommendation 2: Cache Hit Rate
            if ($hit_rate < 50 && $total_cache > 100) {
                $recommendations[] = [
                    'type' => 'warning',
                    'title' => 'Low Cache Hit Rate',
                    'description' => sprintf('Cache hit rate is %.1f%%. Check cache exclusion settings.', $hit_rate),
                    'priority' => 'medium',
                    'action' => 'Review cache configuration'
                ];
            }

            // Recommendation 3: Page Load Time
            if ($load_time > 3) {
                $recommendations[] = [
                    'type' => 'warning',
                    'title' => 'Slow Page Load Time',
                    'description' => sprintf('Average load time is %.2f seconds. Recommended is < 2 seconds.', $load_time),
                    'priority' => 'medium',
                    'action' => 'Enable asset optimization'
                ];
            }

            // Recommendation 4: Stale Optimization
            if ($last_optimization) {
                $age_hours = (time() - $last_optimization) / 3600;
                if ($age_hours > 168) { // 7 days
                    $recommendations[] = [
                        'type' => 'info',
                        'title' => 'Optimization Data Stale',
                        'description' => sprintf('Last optimization was %.0f hours ago. Consider running again.', $age_hours),
                        'priority' => 'low',
                        'action' => 'Run optimization'
                    ];
                }
            } else {
                $recommendations[] = [
                    'type' => 'info',
                    'title' => 'No Optimization Run Yet',
                    'description' => 'Optimization has never been run. Run it to get detailed analysis.',
                    'priority' => 'low',
                    'action' => 'Run initial optimization'
                ];
            }

            return new WP_REST_Response([
                'success' => true,
                'data' => $recommendations,
                'count' => count($recommendations)
            ], 200);
        } catch (Exception $e) {
            return new WP_REST_Response([
                'success' => false,
                'error' => 'Failed to retrieve recommendations'
            ], 500);
        }
    }

    /**
     * Check API permission
     */
    public function check_api_permission() {
        return current_user_can('manage_options');
    }

    /**
     * Handle AJAX API requests
     * Routes to appropriate methods based on endpoint parameter
     */
    public function handle_ajax_api() {
        check_ajax_referer('vssv_ajax_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
        }

        $endpoint = isset($_POST['endpoint']) ? sanitize_text_field($_POST['endpoint']) : '';

        if (empty($endpoint)) {
            wp_send_json_error('No endpoint specified');
        }

        // Route to appropriate handler (stubs for now)
        switch ($endpoint) {
            case 'optimize':
                wp_send_json_success(['message' => 'Optimization feature not yet implemented']);
                break;

            case 'generate_static':
                wp_send_json_success(['message' => 'Static generation feature not yet implemented', 'progress' => 100]);
                break;

            case 'deploy':
                wp_send_json_success(['message' => 'Deploy feature not yet implemented']);
                break;

            default:
                wp_send_json_error('Unknown endpoint: ' . $endpoint);
        }
    }
}
