<?php
/**
 * Cache Management Class
 *
 * Handles page caching, object caching, and cache purging
 *
 * @package VS_Site_Vector
 * @since 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class VSSV_Cache {

    /**
     * Cache settings
     */
    private $settings = [];

    /**
     * Cache statistics
     */
    private $stats = [
        'hits' => 0,
        'misses' => 0,
        'writes' => 0,
        'deletes' => 0
    ];

    /**
     * Constructor
     */
    public function __construct() {
        $this->load_settings();
        $this->init_hooks();
    }

    /**
     * Load cache settings
     */
    private function load_settings() {
        $this->settings = [
            'enabled' => get_option('vssv_cache_enabled', true),
            'ttl' => get_option('vssv_cache_ttl', 3600),
            'exclude_urls' => get_option('vssv_cache_exclude_urls', []),
            'exclude_cookies' => get_option('vssv_cache_exclude_cookies', ['wordpress_logged_in_']),
            'exclude_user_agents' => get_option('vssv_cache_exclude_agents', []),
            'cache_mobile_separately' => get_option('vssv_cache_mobile_separately', true),
            'compression' => get_option('vssv_cache_compression', true),
            'object_cache' => get_option('vssv_object_cache', true),
            'preload' => get_option('vssv_cache_preload', false),
            'browser_cache' => get_option('vssv_browser_cache', true)
        ];
    }

    /**
     * Initialize hooks
     */
    private function init_hooks() {
        if (!$this->settings['enabled']) {
            return;
        }

        // Page caching
        add_action('init', [$this, 'maybe_serve_cached_page'], 0);
        add_action('template_redirect', [$this, 'start_page_cache']);
        add_action('shutdown', [$this, 'maybe_cache_page']);

        // Cache purging
        add_action('save_post', [$this, 'purge_post_cache']);
        add_action('edit_post', [$this, 'purge_post_cache']);
        add_action('delete_post', [$this, 'purge_post_cache']);
        add_action('switch_theme', [$this, 'clear_all']);
        add_action('wp_update_nav_menu', [$this, 'clear_all']);
        add_action('update_option_sidebars_widgets', [$this, 'clear_all']);

        // Object cache
        if ($this->settings['object_cache']) {
            add_filter('pre_transient_', [$this, 'get_object_cache'], 10, 2);
            add_filter('pre_set_transient_', [$this, 'set_object_cache'], 10, 3);
        }

        // AJAX handlers
        add_action('wp_ajax_vssv_purge_cache', [$this, 'ajax_purge_cache']);
        add_action('wp_ajax_vssv_get_cache_stats', [$this, 'ajax_get_cache_stats']);
    }

    /**
     * Maybe serve cached page
     */
    public function maybe_serve_cached_page() {
        // Skip if not cacheable
        if (!$this->is_cacheable_request()) {
            return;
        }

        $cache_key = $this->get_page_cache_key();
        $cached_page = $this->get_cached_page($cache_key);

        if ($cached_page !== false) {
            // Update stats
            $this->stats['hits']++;
            $this->update_stats();

            // Send cache headers
            header('X-VSSV-Cache: HIT');
            header('X-VSSV-Cache-Key: ' . substr(md5($cache_key), 0, 8));
            header('Cache-Control: public, max-age=' . $this->settings['ttl']);
            header('Expires: ' . gmdate('D, d M Y H:i:s', time() + $this->settings['ttl']) . ' GMT');
            header('Last-Modified: ' . gmdate('D, d M Y H:i:s', $this->get_cache_time($cache_key)) . ' GMT');
            header('ETag: ' . substr(md5($cache_key . $this->get_cache_time($cache_key)), 0, 13));

            // Send cached content
            echo $cached_page;

            // Add cache info comment
            if (defined('WP_DEBUG') && WP_DEBUG) {
                echo "\n<!-- Served from VS Site Vector Cache -->";
                echo "\n<!-- Cache generated: " . date('Y-m-d H:i:s', $this->get_cache_time($cache_key)) . " -->";
            }

            exit;
        } else {
            header('X-VSSV-Cache: MISS');
            header('Cache-Control: no-cache, must-revalidate, max-age=0');
            $this->stats['misses']++;
        }
    }

    /**
     * Start page cache capture
     */
    public function start_page_cache() {
        if (!$this->is_cacheable_request()) {
            return;
        }

        ob_start();
    }

    /**
     * Maybe cache the current page
     */
    public function maybe_cache_page() {
        if (!$this->is_cacheable_request()) {
            return;
        }

        $content = ob_get_contents();

        if (!empty($content) && !is_404()) {
            $cache_key = $this->get_page_cache_key();
            $this->cache_page($cache_key, $content);
        }
    }

    /**
     * Check if request is cacheable
     */
    private function is_cacheable_request() {
        // Skip if disabled
        if (!$this->settings['enabled']) {
            return false;
        }

        // Skip admin, AJAX, cron
        if (is_admin() || wp_doing_ajax() || wp_doing_cron()) {
            return false;
        }

        // Skip non-GET requests
        if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
            return false;
        }

        // Skip if logged in
        if (is_user_logged_in()) {
            return false;
        }

        // Check excluded URLs
        $current_url = $_SERVER['REQUEST_URI'];
        foreach ($this->settings['exclude_urls'] as $pattern) {
            // Validate pattern or use simple string matching as fallback
            if (strpos($pattern, '/') === 0) {
                // Looks like a regex pattern, validate it first
                if (@preg_match($pattern, '') === false) {
                    // Invalid regex, fall back to simple string matching
                    if (strpos($current_url, trim($pattern, '/')) !== false) {
                        return false;
                    }
                } else {
                    // Valid regex, use it safely with error suppression
                    if (@preg_match($pattern, $current_url)) {
                        return false;
                    }
                }
            } else {
                // Not a regex, use simple string matching
                if (strpos($current_url, $pattern) !== false) {
                    return false;
                }
            }
        }

        // Check excluded cookies
        foreach ($this->settings['exclude_cookies'] as $cookie) {
            foreach ($_COOKIE as $key => $value) {
                if (strpos($key, $cookie) !== false) {
                    return false;
                }
            }
        }

        // Check excluded user agents
        if (!empty($_SERVER['HTTP_USER_AGENT'])) {
            foreach ($this->settings['exclude_user_agents'] as $agent) {
                if (strpos($_SERVER['HTTP_USER_AGENT'], $agent) !== false) {
                    return false;
                }
            }
        }

        // Check query strings
        if (!empty($_GET) && !isset($_GET['utm_source'])) {
            return false;
        }

        return true;
    }

    /**
     * Get page cache key
     */
    private function get_page_cache_key() {
        $key_parts = [
            'page',
            $_SERVER['HTTP_HOST'] ?? parse_url(home_url(), PHP_URL_HOST) ?? 'localhost',
            $_SERVER['REQUEST_URI'] ?? '/'
        ];

        // Add mobile suffix if caching separately
        if ($this->settings['cache_mobile_separately'] && wp_is_mobile()) {
            $key_parts[] = 'mobile';
        }

        // Add protocol
        $key_parts[] = is_ssl() ? 'https' : 'http';

        // Add language if WPML/Polylang
        if (defined('ICL_LANGUAGE_CODE')) {
            $key_parts[] = ICL_LANGUAGE_CODE;
        }

        return 'vssv_cache_' . md5(implode('_', $key_parts));
    }

    /**
     * Get cached page
     */
    private function get_cached_page($cache_key) {
        $cache_file = VSSV_CACHE_DIR . 'html/' . $cache_key . '.html';

        // Validate file path to prevent path traversal
        if (!$this->validate_cache_file_path($cache_file)) {
            return false;
        }

        if (!file_exists($cache_file)) {
            return false;
        }

        // Check if cache is expired
        if ((time() - filemtime($cache_file)) > $this->settings['ttl']) {
            if (file_exists($cache_file)) {
                unlink($cache_file);
            }
            return false;
        }

        // Read cache file
        $content = file_get_contents($cache_file);

        // Decompress if needed
        if ($this->settings['compression'] && function_exists('gzuncompress')) {
            $content = gzuncompress($content);
        }

        return $content;
    }

    /**
     * Cache page content
     */
    private function cache_page($cache_key, $content) {
        $cache_file = VSSV_CACHE_DIR . 'html/' . $cache_key . '.html';

        // Compress if enabled
        if ($this->settings['compression'] && function_exists('gzcompress')) {
            $content = gzcompress($content, 9);
        }

        // Validate file path before writing
        if (!$this->validate_cache_file_path($cache_file)) {
            return false;
        }

        // Write cache file
        file_put_contents($cache_file, $content, LOCK_EX);

        // Update stats
        $this->stats['writes']++;
        $this->update_stats();

        // Store metadata
        $meta = [
            'url' => $_SERVER['REQUEST_URI'],
            'time' => time(),
            'size' => strlen($content),
            'compressed' => $this->settings['compression']
        ];

        update_option('vssv_cache_meta_' . $cache_key, $meta);
    }

    /**
     * Get cache time
     */
    private function get_cache_time($cache_key) {
        $cache_file = VSSV_CACHE_DIR . 'html/' . $cache_key . '.html';
        return file_exists($cache_file) ? filemtime($cache_file) : 0;
    }

    /**
     * Purge post cache
     */
    public function purge_post_cache($post_id) {
        // Clear post cache
        $this->clear_url_cache(get_permalink($post_id));

        // Clear home page
        $this->clear_url_cache(home_url('/'));

        // Clear archive pages
        $post = get_post($post_id);
        if ($post) {
            // Clear category archives
            $categories = wp_get_post_categories($post_id);
            foreach ($categories as $cat_id) {
                $this->clear_url_cache(get_category_link($cat_id));
            }

            // Clear tag archives
            $tags = wp_get_post_tags($post_id);
            foreach ($tags as $tag) {
                $this->clear_url_cache(get_tag_link($tag->term_id));
            }

            // Clear author archive
            $this->clear_url_cache(get_author_posts_url($post->post_author));
        }

        // Clear related caches
        do_action('vssv_purge_post_cache', $post_id);
    }

    /**
     * Clear URL cache
     */
    private function clear_url_cache($url) {
        $parsed = parse_url($url);
        $cache_key = $this->get_page_cache_key_for_url($parsed['path']);

        $cache_file = VSSV_CACHE_DIR . 'html/' . $cache_key . '.html';
        if (file_exists($cache_file)) {
            unlink($cache_file);
            $this->stats['deletes']++;
        }

        // Clear variants (mobile, etc.)
        $variants = glob(VSSV_CACHE_DIR . 'html/' . $cache_key . '_*.html');
        if ($variants) {
            foreach ($variants as $variant) {
                unlink($variant);
                $this->stats['deletes']++;
            }
        }
    }

    /**
     * Get cache key for specific URL
     */
    private function get_page_cache_key_for_url($path) {
        $key_parts = [
            'page',
            $_SERVER['HTTP_HOST'],
            $path
        ];

        return 'vssv_cache_' . md5(implode('_', $key_parts));
    }

    /**
     * Clear all caches
     */
    public function clear_all() {
        // Clear HTML cache
        $files = glob(VSSV_CACHE_DIR . 'html/*.html');
        if ($files) {
            foreach ($files as $file) {
                unlink($file);
            }
        }

        // Clear CSS cache
        $files = glob(VSSV_CACHE_DIR . 'css/*.css');
        if ($files) {
            foreach ($files as $file) {
                unlink($file);
            }
        }

        // Clear JS cache
        $files = glob(VSSV_CACHE_DIR . 'js/*.js');
        if ($files) {
            foreach ($files as $file) {
                unlink($file);
            }
        }

        // Clear critical CSS
        $files = glob(VSSV_CACHE_DIR . 'critical/*.css');
        if ($files) {
            foreach ($files as $file) {
                unlink($file);
            }
        }

        // Clear fragment cache
        $files = glob(VSSV_CACHE_DIR . 'fragments/*.html');
        if ($files) {
            foreach ($files as $file) {
                unlink($file);
            }
        }

        // Clear object cache
        wp_cache_flush();

        // Clear database cache metadata
        global $wpdb;
        $wpdb->query(
            $wpdb->prepare(
                "DELETE FROM {$wpdb->options} WHERE option_name LIKE %s",
                $wpdb->esc_like('vssv_cache_meta_') . '%'
            )
        );

        // Reset stats
        $this->stats = [
            'hits' => 0,
            'misses' => 0,
            'writes' => 0,
            'deletes' => 0
        ];
        $this->update_stats();

        // Trigger action
        do_action('vssv_cache_cleared_all');

        return true;
    }

    /**
     * Clear expired caches
     */
    public function clear_expired() {
        $ttl = $this->settings['ttl'];
        $now = time();
        $count = 0;

        // Clear expired HTML cache
        $files = glob(VSSV_CACHE_DIR . 'html/*.html');
        if ($files) {
            foreach ($files as $file) {
                if (($now - filemtime($file)) > $ttl) {
                    unlink($file);
                    $count++;
                }
            }
        }

        // Clear expired fragments
        $files = glob(VSSV_CACHE_DIR . 'fragments/*.html');
        if ($files) {
            foreach ($files as $file) {
                if (($now - filemtime($file)) > 3600) { // 1 hour for fragments
                    unlink($file);
                    $count++;
                }
            }
        }

        return $count;
    }

    /**
     * Rebuild cache
     */
    public function rebuild() {
        // Clear all
        $this->clear_all();

        return true;
    }

    /**
     * Update cache statistics
     */
    private function update_stats() {
        update_option('vssv_cache_stats', $this->stats);

        // Calculate hit rate
        $total = $this->stats['hits'] + $this->stats['misses'];
        if ($total > 0) {
            $hit_rate = ($this->stats['hits'] / $total) * 100;
            set_transient('vssv_cache_hit_rate', $hit_rate, HOUR_IN_SECONDS);
        }
    }

    /**
     * Get object cache
     */
    public function get_object_cache($pre, $transient) {
        if (!$this->settings['object_cache']) {
            return $pre;
        }

        // Check persistent object cache
        if (function_exists('wp_cache_get')) {
            $value = wp_cache_get($transient, 'vssv_transients');
            if ($value !== false) {
                $this->stats['hits']++;
                return $value;
            }
        }

        $this->stats['misses']++;
        return $pre;
    }

    /**
     * Set object cache
     */
    public function set_object_cache($pre, $transient, $value) {
        if (!$this->settings['object_cache']) {
            return $pre;
        }

        // Store in persistent object cache
        if (function_exists('wp_cache_set')) {
            wp_cache_set($transient, $value, 'vssv_transients', $this->settings['ttl']);
            $this->stats['writes']++;
        }

        return $pre;
    }

    /**
     * AJAX purge cache handler
     */
    public function ajax_purge_cache() {
        check_ajax_referer('vssv_ajax_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }

        $type = sanitize_text_field($_POST['type'] ?? 'all');

        if ($type === 'all') {
            $this->clear_all();
            $message = 'All caches cleared';
        } elseif ($type === 'expired') {
            $count = $this->clear_expired();
            $message = sprintf('%d expired cache files cleared', $count);
        } elseif ($type === 'page' && !empty($_POST['url'])) {
            $this->clear_url_cache(esc_url_raw($_POST['url']));
            $message = 'Page cache cleared';
        }

        wp_send_json_success(['message' => $message]);
    }

    /**
     * AJAX get cache stats
     */
    public function ajax_get_cache_stats() {
        // Security: Check user has manage_options capability
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Permission denied.']);
        }

        $stats = get_option('vssv_cache_stats', $this->stats);
        $hit_rate = get_transient('vssv_cache_hit_rate') ?: 0;

        // Get cache size
        $cache_size = 0;
        $cache_files = 0;

        $directories = ['html', 'css', 'js', 'fragments', 'critical'];
        foreach ($directories as $dir) {
            $files = glob(VSSV_CACHE_DIR . $dir . '/*');
            if ($files) {
                $cache_files += count($files);
                foreach ($files as $file) {
                    $cache_size += filesize($file);
                }
            }
        }

        wp_send_json_success([
            'stats' => $stats,
            'hit_rate' => $hit_rate,
            'cache_size' => size_format($cache_size),
            'cache_files' => $cache_files
        ]);
    }

    /**
     * Run optimization
     */
    public function optimize() {
        // Clear expired caches
        $expired = $this->clear_expired();

        // Optimize cache storage
        $this->optimize_storage();

        // Preload important pages
        if ($this->settings['preload']) {
            $this->preload_cache();
        }

        return [
            'expired_cleared' => $expired,
            'cache_optimized' => true,
            'preloaded' => $this->settings['preload']
        ];
    }

    /**
     * Optimize cache storage
     */
    private function optimize_storage() {
        // Remove orphaned metadata
        global $wpdb;
        $wpdb->query($wpdb->prepare(
            "DELETE FROM {$wpdb->options}
            WHERE option_name LIKE %s
            AND option_value < %d",
            'vssv_cache_meta_%',
            (time() - $this->settings['ttl'])
        ));
    }

    /**
     * Preload cache for important pages
     */
    private function preload_cache() {
        // Get list of pages to preload (homepage, popular posts, key pages)
        $preload_urls = [];

        // Always preload homepage
        $preload_urls[] = home_url('/');

        // Get popular posts
        $popular_posts = get_posts([
            'posts_per_page' => 5,
            'orderby' => 'comment_count',
            'order' => 'DESC'
        ]);

        foreach ($popular_posts as $post) {
            $preload_urls[] = get_permalink($post->ID);
        }

        // Preload each URL
        foreach ($preload_urls as $url) {
            wp_remote_get($url, [
                'blocking' => false,
                'sslverify' => apply_filters('https_local_ssl_verify', false)
            ]);
        }

        set_transient('vssv_cache_preload_complete', true, HOUR_IN_SECONDS);
    }

    /**
     * Get recommendations
     */
    public function get_recommendations() {
        $recommendations = [];

        $hit_rate = get_transient('vssv_cache_hit_rate') ?: 0;

        if ($hit_rate < 50) {
            $recommendations[] = [
                'type' => 'warning',
                'message' => sprintf('Low cache hit rate: %.1f%%', $hit_rate),
                'action' => 'Review cache exclusions and TTL settings'
            ];
        }

        if (!$this->settings['browser_cache']) {
            $recommendations[] = [
                'type' => 'info',
                'message' => 'Browser caching is disabled',
                'action' => 'Enable browser caching for better performance'
            ];
        }

        if (!$this->settings['compression']) {
            $recommendations[] = [
                'type' => 'info',
                'message' => 'Cache compression is disabled',
                'action' => 'Enable compression to reduce cache size'
            ];
        }

        return $recommendations;
    }

    /**
     * Validate cache file path to prevent path traversal attacks
     */
    private function validate_cache_file_path($file_path) {
        // Define allowed cache directories
        $allowed_dirs = [
            VSSV_CACHE_DIR . 'html/',
            VSSV_CACHE_DIR . 'css/',
            VSSV_CACHE_DIR . 'js/',
        ];

        // Get real path and resolve symbolic links
        $real_file_path = realpath($file_path);
        if ($real_file_path === false) {
            return false;
        }

        // Check if the path is within allowed directories
        foreach ($allowed_dirs as $allowed_dir) {
            $real_allowed_dir = realpath($allowed_dir);
            if ($real_allowed_dir !== false && strpos($real_file_path, $real_allowed_dir) === 0) {
                // Additional check: ensure it's a regular file
                if (file_exists($real_file_path) && is_file($real_file_path)) {
                    return true;
                }
                return true; // File doesn't exist yet, but path is valid
            }
        }

        return false;
    }
}