<?php
/**
 * VS Site Vector Configuration
 *
 * Provides safe defaults and critical settings
 *
 * @package VS_Site_Vector
 * @since 1.1.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class VSSV_Config {

    /**
     * Get default configuration
     */
    public static function get_defaults() {
        return [
            // Cache Settings
            'vssv_cache_enabled' => true,
            'vssv_cache_ttl' => 3600,
            'vssv_cache_mobile_separately' => true,

            // Critical Exclusions - NEVER cache these
            'vssv_cache_exclude_urls' => [
                '/wp-admin/*',
                '/wp-login.php',
                '/wp-cron.php',
                '*/feed/*',
                '*/cart/*',
                '*/checkout/*',
                '*/my-account/*',
                '*/addons/*',
                '*.xml',
                '*.txt',
                '/wp-json/*'
            ],

            // Cookie exclusions - bypass cache for these
            'vssv_cache_exclude_cookies' => [
                'wordpress_logged_in_',
                'wordpress_sec_',
                'wp-settings-',
                'wp-postpass_',
                'woocommerce_items_in_cart',
                'woocommerce_cart_hash',
                'edd_items_in_cart',
                'it_exchange_session_'
            ],

            // Query string exclusions
            'vssv_cache_exclude_query_strings' => [
                'utm_source',
                'utm_medium',
                'utm_campaign',
                'utm_term',
                'utm_content',
                'fbclid',
                'gclid',
                'preview',
                'preview_id',
                'preview_nonce',
                'elementor-preview',
                's', // Search query
                'p', // Preview
            ],

            // User agent exclusions (bots, crawlers)
            'vssv_cache_exclude_agents' => [
                'bot',
                'crawler',
                'spider',
                'curl',
                'wget',
                'facebookexternalhit',
                'WhatsApp',
                'Telegram'
            ],

            // HTML Optimization (safe)
            'vssv_minify_html' => true,
            'vssv_remove_html_comments' => true,
            'vssv_remove_generator_tags' => true,

            // CSS Optimization (test carefully)
            'vssv_minify_css' => true,
            'vssv_combine_css' => false, // Can break with HTTP/2
            'vssv_critical_css' => false, // Requires setup
            'vssv_remove_unused_css' => false, // Very risky

            // JS Optimization (often problematic)
            'vssv_minify_js' => false, // Often breaks modern JS
            'vssv_combine_js' => false, // Can break dependencies
            'vssv_defer_js' => false, // Test thoroughly
            'vssv_delay_js' => false, // Can break functionality

            // Image Optimization (generally safe)
            'vssv_lazy_load_images' => true,
            'vssv_lazy_load_iframes' => true,
            'vssv_lazy_load_videos' => true,
            'vssv_add_missing_image_dimensions' => true,
            'vssv_webp_conversion' => false, // Requires server support

            // Database Optimization
            'vssv_limit_post_revisions' => 5,
            'vssv_optimize_database_daily' => true,
            'vssv_clean_expired_transients' => true,
            'vssv_clean_spam_comments' => true,
            'vssv_clean_trash_posts' => true,
            'vssv_clean_orphaned_meta' => false, // Risky

            // WordPress Optimization
            'vssv_disable_emojis' => true,
            'vssv_disable_embeds' => false, // Breaks oEmbed
            'vssv_disable_xmlrpc' => false, // May break apps
            'vssv_remove_query_strings' => true,
            'vssv_disable_heartbeat' => false, // Affects autosave
            'vssv_limit_heartbeat' => 60, // Seconds

            // CDN Settings
            'vssv_cdn_enabled' => false,
            'vssv_cdn_url' => '',
            'vssv_cdn_include_types' => ['css', 'js', 'images'],
            'vssv_cdn_exclude_files' => [],

            // Advanced Settings
            'vssv_enable_object_cache' => false, // Requires Redis/Memcached
            'vssv_enable_browser_cache' => true,
            'vssv_enable_gzip_compression' => true,
            'vssv_preload_cache' => false, // Resource intensive
            'vssv_warmup_cache_on_save' => true,

            // Debug & Monitoring
            'vssv_debug_mode' => false,
            'vssv_log_errors' => true,
            'vssv_show_admin_bar_stats' => true,
        ];
    }

    /**
     * Get critical settings that should not be changed in production
     */
    public static function get_critical_settings() {
        return [
            'exclude_urls' => [
                '/wp-admin/*',
                '/wp-login.php',
                '/wp-json/*',
                '*/cart/*',
                '*/checkout/*'
            ],
            'exclude_cookies' => [
                'wordpress_logged_in_',
                'woocommerce_items_in_cart'
            ],
            'max_cache_size' => 1073741824, // 1GB
            'max_file_size' => 5242880, // 5MB
        ];
    }

    /**
     * Get environment-specific settings
     */
    public static function get_environment_settings($environment = 'production') {
        $settings = [];

        switch ($environment) {
            case 'development':
                $settings = [
                    'vssv_cache_enabled' => false,
                    'vssv_minify_html' => false,
                    'vssv_minify_css' => false,
                    'vssv_minify_js' => false,
                    'vssv_debug_mode' => true,
                    'vssv_cache_ttl' => 0,
                ];
                break;

            case 'staging':
                $settings = [
                    'vssv_cache_enabled' => true,
                    'vssv_cache_ttl' => 300, // 5 minutes
                    'vssv_minify_html' => true,
                    'vssv_minify_css' => true,
                    'vssv_minify_js' => false,
                    'vssv_debug_mode' => true,
                ];
                break;

            case 'production':
            default:
                $settings = [
                    'vssv_cache_enabled' => true,
                    'vssv_cache_ttl' => 3600,
                    'vssv_minify_html' => true,
                    'vssv_minify_css' => true,
                    'vssv_minify_js' => false, // Enable only after testing
                    'vssv_debug_mode' => false,
                ];
                break;
        }

        return $settings;
    }

    /**
     * Get compatibility settings for popular plugins
     */
    public static function get_compatibility_settings() {
        $settings = [];

        // WooCommerce
        if (class_exists('WooCommerce')) {
            $settings['woocommerce'] = [
                'exclude_pages' => [
                    wc_get_page_id('cart'),
                    wc_get_page_id('checkout'),
                    wc_get_page_id('myaccount'),
                ],
                'exclude_cookies' => [
                    'woocommerce_items_in_cart',
                    'woocommerce_cart_hash',
                    'woocommerce_recently_viewed',
                    'wp_woocommerce_session_',
                ],
            ];
        }

        // Elementor
        if (defined('ELEMENTOR_VERSION')) {
            $settings['elementor'] = [
                'exclude_urls' => [
                    '*elementor-preview=*',
                    '*action=elementor*',
                ],
                'clear_cache_on_save' => true,
            ];
        }

        // WPML
        if (defined('ICL_SITEPRESS_VERSION')) {
            $settings['wpml'] = [
                'cache_by_language' => true,
                'exclude_cookies' => [
                    'wpml_browser_redirect_test',
                    '_icl_current_language',
                ],
            ];
        }

        // Contact Form 7
        if (defined('WPCF7_VERSION')) {
            $settings['cf7'] = [
                'exclude_pages_with_form' => true,
                'ajax_optimization' => false,
            ];
        }

        // Gravity Forms
        if (class_exists('GFForms')) {
            $settings['gravity_forms'] = [
                'exclude_pages_with_form' => true,
                'exclude_query_strings' => ['gf_page'],
            ];
        }

        return $settings;
    }

    /**
     * Validate settings before saving
     */
    public static function validate_settings($settings) {
        $errors = [];

        // Check cache TTL
        if (isset($settings['vssv_cache_ttl'])) {
            $ttl = intval($settings['vssv_cache_ttl']);
            if ($ttl < 0 || $ttl > 86400) {
                $errors[] = 'Cache TTL must be between 0 and 86400 seconds';
            }
        }

        // Check CDN URL format
        if (!empty($settings['vssv_cdn_url'])) {
            if (!filter_var($settings['vssv_cdn_url'], FILTER_VALIDATE_URL)) {
                $errors[] = 'CDN URL must be a valid URL';
            }
        }

        // Check for conflicting settings
        if ($settings['vssv_combine_css'] && $settings['vssv_critical_css']) {
            $errors[] = 'Cannot use combined CSS with critical CSS';
        }

        if ($settings['vssv_combine_js'] && $settings['vssv_defer_js']) {
            $errors[] = 'Combined JS may conflict with deferred loading';
        }

        return $errors;
    }

    /**
     * Get recommended settings based on site analysis
     */
    public static function get_recommended_settings() {
        global $wp_version;
        $recommendations = [];

        // PHP version check
        if (version_compare(PHP_VERSION, '7.4', '>=')) {
            $recommendations['vssv_enable_opcache'] = true;
        }

        // WordPress version check
        if (version_compare($wp_version, '5.5', '>=')) {
            $recommendations['vssv_lazy_load_images'] = true; // Native lazy loading
        }

        // Server check
        if (function_exists('gzencode')) {
            $recommendations['vssv_enable_gzip_compression'] = true;
        }

        // Check for Redis/Memcached
        if (class_exists('Redis') || class_exists('Memcached')) {
            $recommendations['vssv_enable_object_cache'] = true;
        }

        // Check server software
        $server = $_SERVER['SERVER_SOFTWARE'] ?? '';
        if (strpos($server, 'nginx') !== false) {
            $recommendations['vssv_server_type'] = 'nginx';
        } elseif (strpos($server, 'Apache') !== false) {
            $recommendations['vssv_server_type'] = 'apache';
        }

        return $recommendations;
    }

    /**
     * Emergency disable check
     */
    public static function is_emergency_disabled() {
        return defined('VSSV_EMERGENCY_DISABLE') && VSSV_EMERGENCY_DISABLE === true;
    }

    /**
     * Check if running in safe mode
     */
    public static function is_safe_mode() {
        return (
            isset($_GET['vssv_safe_mode']) &&
            $_GET['vssv_safe_mode'] === '1' &&
            current_user_can('manage_options')
        );
    }
}