<?php
/**
 * Deployment Management Module
 *
 * Handles deployments to various CDN and hosting platforms
 *
 * @package VS_Site_Vector
 * @since 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class VSSV_Deploy {

    /**
     * Deployment settings
     */
    private $settings = [];

    /**
     * Supported platforms
     */
    private $platforms = [
        'cloudflare' => 'Cloudflare Pages'
    ];

    /**
     * Constructor
     */
    public function __construct() {
        $this->load_settings();
        $this->init_hooks();
    }

    /**
     * Load settings
     */
    private function load_settings() {
        $this->settings = [
            'enabled' => get_option('vssv_deploy_enabled', false),
            'primary_platform' => get_option('vssv_deploy_primary', 'cloudflare'),
            'multi_deploy' => get_option('vssv_deploy_multi', false),
            'platforms' => $this->load_platform_settings(),
            'auto_deploy' => get_option('vssv_auto_deploy', true),
            'deploy_on_save' => get_option('vssv_deploy_on_save', true),
            'deploy_branch' => get_option('vssv_deploy_branch', 'main'),
            'preview_deploys' => get_option('vssv_preview_deploys', true),
            'rollback_enabled' => get_option('vssv_rollback_enabled', true),
            'notifications' => get_option('vssv_deploy_notifications', true),
            'slack_webhook' => get_option('vssv_slack_webhook', ''),
            'deploy_hooks' => get_option('vssv_deploy_hooks', [])
        ];
    }

    /**
     * Load platform-specific settings
     */
    private function load_platform_settings() {
        $platforms = [];

        // Cloudflare Pages
        $platforms['cloudflare'] = [
            'enabled' => get_option('vssv_cloudflare_enabled', false),
            'account_id' => get_option('vssv_cloudflare_account_id', ''),
            'api_token' => get_option('vssv_cloudflare_api_token', ''),
            'project_name' => get_option('vssv_cloudflare_project', ''),
            'webhook_url' => get_option('vssv_cloudflare_webhook', ''),
            'production_branch' => get_option('vssv_cloudflare_branch', 'main'),
            'build_command' => get_option('vssv_cloudflare_build_cmd', 'php generate-static.php'),
            'output_directory' => get_option('vssv_cloudflare_output', 'static-output'),
            'environment_variables' => get_option('vssv_cloudflare_env', [])
        ];

        return $platforms;
    }

    /**
     * Initialize hooks
     */
    private function init_hooks() {
        if (!$this->settings['enabled']) {
            return;
        }

        // Deploy actions
        add_action('vssv_deploy_requested', [$this, 'deploy']);

        // AJAX handlers
        add_action('wp_ajax_vssv_deploy', [$this, 'ajax_deploy']);
        add_action('wp_ajax_vssv_deploy_status', [$this, 'ajax_get_status']);
        add_action('wp_ajax_vssv_deploy_rollback', [$this, 'ajax_rollback']);
        add_action('wp_ajax_vssv_test_deployment', [$this, 'ajax_test_deployment']);

        // Admin UI
        add_action('admin_menu', [$this, 'add_admin_menu']);

        // REST API
        add_action('rest_api_init', [$this, 'register_rest_routes']);

        // Webhook receivers
        add_action('init', [$this, 'register_webhook_endpoints']);
    }

    /**
     * Deploy to configured platforms
     */
    public function deploy($args = []) {
        $defaults = [
            'platforms' => [$this->settings['primary_platform']],
            'type' => 'production',
            'message' => 'Manual deployment from VS Site Vector',
            'files' => [],
            'branch' => $this->settings['deploy_branch']
        ];
        $args = wp_parse_args($args, $defaults);

        $results = [];

        // Deploy to each platform
        foreach ($args['platforms'] as $platform) {
            if (!$this->is_platform_configured($platform)) {
                $results[$platform] = [
                    'success' => false,
                    'error' => 'Platform not configured'
                ];
                continue;
            }

            $method = 'deploy_to_' . $platform;
            if (method_exists($this, $method)) {
                $results[$platform] = $this->$method($args);
            } else {
                $results[$platform] = $this->deploy_via_webhook($platform, $args);
            }
        }

        // Send notifications
        if ($this->settings['notifications']) {
            $this->send_deployment_notification($results, $args);
        }

        // Log deployment
        $this->log_deployment($results, $args);

        return $results;
    }

    /**
     * Deploy to Cloudflare Pages
     */
    private function deploy_to_cloudflare($args) {
        $config = $this->settings['platforms']['cloudflare'];

        // Method 1: Webhook
        if (!empty($config['webhook_url'])) {
            return $this->deploy_via_webhook('cloudflare', $args);
        }

        // Method 2: Direct API
        if (!empty($config['api_token']) && !empty($config['account_id'])) {
            return $this->deploy_via_cloudflare_api($args);
        }

        return [
            'success' => false,
            'error' => 'No deployment method configured for Cloudflare'
        ];
    }

    /**
     * Deploy via Cloudflare API
     */
    private function deploy_via_cloudflare_api($args) {
        $config = $this->settings['platforms']['cloudflare'];

        // Prepare deployment
        $endpoint = sprintf(
            'https://api.cloudflare.com/client/v4/accounts/%s/pages/projects/%s/deployments',
            $config['account_id'],
            $config['project_name']
        );

        // Create deployment
        $response = wp_remote_post($endpoint, [
            'headers' => [
                'Authorization' => 'Bearer ' . $config['api_token'],
                'Content-Type' => 'application/json'
            ],
            'body' => json_encode([
                'branch' => $args['branch'],
                'deployment_trigger' => [
                    'type' => 'ad_hoc',
                    'metadata' => [
                        'source' => 'vs_site_vector',
                        'message' => $args['message']
                    ]
                ]
            ])
        ]);

        if (is_wp_error($response)) {
            return [
                'success' => false,
                'error' => $response->get_error_message()
            ];
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);

        if ($body['success']) {
            return [
                'success' => true,
                'deployment_id' => $body['result']['id'],
                'url' => $body['result']['url'],
                'preview_url' => $body['result']['preview_url']
            ];
        }

        return [
            'success' => false,
            'error' => $body['errors'][0]['message'] ?? 'Unknown error'
        ];
    }

    /**
     * Deploy via webhook
     */
    private function deploy_via_webhook($platform, $args) {
        $config = $this->settings['platforms'][$platform];

        if (empty($config['webhook_url'])) {
            return [
                'success' => false,
                'error' => 'Webhook URL not configured'
            ];
        }

        $response = wp_remote_post($config['webhook_url'], [
            'timeout' => 10,
            'body' => [
                'platform' => $platform,
                'type' => $args['type'],
                'branch' => $args['branch'],
                'message' => $args['message'],
                'source' => 'vs_site_vector',
                'site_url' => home_url()
            ]
        ]);

        if (is_wp_error($response)) {
            return [
                'success' => false,
                'error' => $response->get_error_message()
            ];
        }

        $code = wp_remote_retrieve_response_code($response);

        if ($code >= 200 && $code < 300) {
            return [
                'success' => true,
                'message' => 'Deployment webhook triggered successfully',
                'status_code' => $code
            ];
        }

        return [
            'success' => false,
            'error' => 'Webhook returned status code: ' . $code
        ];
    }

    /**
     * Create deployment ZIP
     */
    private function create_deployment_zip($source_dir) {
        if (!class_exists('ZipArchive')) {
            return false;
        }

        $zip_file = wp_tempnam('vssv_deploy_');
        $zip = new ZipArchive();

        if ($zip->open($zip_file, ZipArchive::CREATE) !== TRUE) {
            return false;
        }

        // Add files recursively
        $this->add_files_to_zip($zip, $source_dir, '');

        $zip->close();

        return $zip_file;
    }

    /**
     * Add files to ZIP recursively
     */
    private function add_files_to_zip($zip, $dir, $base) {
        $files = scandir($dir);

        foreach ($files as $file) {
            if ($file === '.' || $file === '..') {
                continue;
            }

            $path = $dir . '/' . $file;
            $zip_path = $base ? $base . '/' . $file : $file;

            if (is_dir($path)) {
                $zip->addEmptyDir($zip_path);
                $this->add_files_to_zip($zip, $path, $zip_path);
            } else {
                $zip->addFile($path, $zip_path);
            }
        }
    }

    /**
     * Check if platform is configured
     */
    private function is_platform_configured($platform) {
        if (!isset($this->settings['platforms'][$platform])) {
            return false;
        }

        $config = $this->settings['platforms'][$platform];

        if (!$config['enabled']) {
            return false;
        }

        // Check for required configuration
        if ($platform === 'cloudflare') {
            return !empty($config['webhook_url']) ||
                   (!empty($config['api_token']) && !empty($config['account_id']));
        }

        return false;
    }

    /**
     * Send deployment notification
     */
    private function send_deployment_notification($results, $args) {
        // Prepare notification message
        $successful = array_filter($results, function($r) { return $r['success']; });
        $failed = array_filter($results, function($r) { return !$r['success']; });

        $message = sprintf(
            "Deployment %s\n",
            empty($failed) ? 'Successful' : 'Completed with Errors'
        );

        if (!empty($successful)) {
            $message .= "\nSuccessful deployments:\n";
            foreach ($successful as $platform => $result) {
                $message .= "- {$this->platforms[$platform]}\n";
                if (!empty($result['url'])) {
                    $message .= "  URL: {$result['url']}\n";
                }
            }
        }

        if (!empty($failed)) {
            $message .= "\nFailed deployments:\n";
            foreach ($failed as $platform => $result) {
                $message .= "- {$this->platforms[$platform]}: {$result['error']}\n";
            }
        }

        // Send to Slack if configured
        if (!empty($this->settings['slack_webhook'])) {
            wp_remote_post($this->settings['slack_webhook'], [
                'body' => json_encode([
                    'text' => $message,
                    'username' => 'VS Site Vector',
                    'icon_emoji' => ':rocket:'
                ])
            ]);
        }

        // Send email to admin
        $admin_email = get_option('admin_email');
        wp_mail(
            $admin_email,
            'VS Site Vector Deployment Notification',
            $message
        );
    }

    /**
     * Log deployment
     */
    private function log_deployment($results, $args) {
        global $wpdb;

        $wpdb->insert(
            $wpdb->prefix . 'vssv_optimization_log',
            [
                'optimization_type' => 'deployment',
                'before_value' => json_encode($args),
                'after_value' => json_encode($results),
                'timestamp' => current_time('mysql')
            ]
        );

        // Store last deployment info
        update_option('vssv_last_deployment', [
            'time' => current_time('timestamp'),
            'results' => $results,
            'args' => $args
        ]);
    }

    /**
     * Add admin menu
     */
    public function add_admin_menu() {
        add_submenu_page(
            'vs-site-vector',
            'Deployment',
            'Deployment',
            'manage_options',
            'vs-site-vector-deploy',
            [$this, 'render_deploy_page']
        );
    }

    /**
     * Render deployment page
     */
    public function render_deploy_page() {
        // This would render the deployment configuration UI
        include VSSV_PLUGIN_DIR . 'admin/views/deploy-page.php';
    }

    /**
     * Register REST routes
     */
    public function register_rest_routes() {
        register_rest_route('vssv/v1', '/deploy', [
            'methods' => 'POST',
            'callback' => [$this, 'rest_deploy'],
            'permission_callback' => function() {
                return current_user_can('manage_options');
            }
        ]);

        register_rest_route('vssv/v1', '/deploy/status/(?P<platform>[a-z]+)', [
            'methods' => 'GET',
            'callback' => [$this, 'rest_get_status'],
            'permission_callback' => function() {
                return current_user_can('manage_options');
            }
        ]);

        register_rest_route('vssv/v1', '/deploy/rollback/(?P<platform>[a-z]+)', [
            'methods' => 'POST',
            'callback' => [$this, 'rest_rollback'],
            'permission_callback' => function() {
                return current_user_can('manage_options');
            }
        ]);
    }

    /**
     * REST: Deploy
     */
    public function rest_deploy($request) {
        $params = $request->get_params();

        $result = $this->deploy([
            'platforms' => $params['platforms'] ?? [$this->settings['primary_platform']],
            'type' => $params['type'] ?? 'production',
            'message' => $params['message'] ?? 'API deployment'
        ]);

        return new WP_REST_Response($result);
    }

    /**
     * REST: Get deployment status
     */
    public function rest_get_status($request) {
        $platform = $request->get_param('platform');

        // Get status based on platform
        $status = $this->get_deployment_status($platform);

        return new WP_REST_Response($status);
    }

    /**
     * Get deployment status
     */
    private function get_deployment_status($platform) {
        // This would check the actual deployment status
        // For now, return last deployment info
        $last = get_option('vssv_last_deployment');

        if ($last && isset($last['results'][$platform])) {
            return [
                'platform' => $platform,
                'last_deployment' => $last['time'],
                'status' => $last['results'][$platform]['success'] ? 'success' : 'failed',
                'details' => $last['results'][$platform]
            ];
        }

        return [
            'platform' => $platform,
            'status' => 'no_deployments'
        ];
    }

    /**
     * Register webhook endpoints
     */
    public function register_webhook_endpoints() {
        // Handle deployment webhooks from platforms
        add_rewrite_rule(
            '^vssv-deploy-webhook/([a-z]+)/?',
            'index.php?vssv_deploy_webhook=$matches[1]',
            'top'
        );

        add_filter('query_vars', function($vars) {
            $vars[] = 'vssv_deploy_webhook';
            return $vars;
        });

        add_action('template_redirect', [$this, 'handle_webhook']);
    }

    /**
     * Handle incoming webhooks
     */
    public function handle_webhook() {
        $platform = get_query_var('vssv_deploy_webhook');

        if (!$platform) {
            return;
        }

        // Verify webhook signature if applicable
        $body = file_get_contents('php://input');
        $data = json_decode($body, true);

        // Process webhook based on platform
        if ($platform === 'cloudflare') {
            $this->handle_cloudflare_webhook($data);
        }

        // Send response
        wp_send_json_success(['received' => true]);
        exit;
    }

    /**
     * Handle Cloudflare webhook
     */
    private function handle_cloudflare_webhook($data) {
        // Process Cloudflare deployment notification
        if ($data['event'] === 'deployment.created') {
            update_option('vssv_cloudflare_last_deployment', [
                'id' => $data['deployment']['id'],
                'url' => $data['deployment']['url'],
                'time' => current_time('timestamp')
            ]);
        }
    }

    /**
     * AJAX handlers
     */
    public function ajax_deploy() {
        check_ajax_referer('vssv_ajax_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }

        $platforms = $_POST['platforms'] ?? [$this->settings['primary_platform']];
        $type = $_POST['type'] ?? 'production';

        $result = $this->deploy([
            'platforms' => $platforms,
            'type' => $type,
            'message' => 'Manual deployment from admin'
        ]);

        wp_send_json_success($result);
    }

    /**
     * AJAX: Get deployment status
     */
    public function ajax_get_status() {
        check_ajax_referer('vssv_ajax_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }

        $platform = $_POST['platform'] ?? $this->settings['primary_platform'];
        $status = $this->get_deployment_status($platform);

        wp_send_json_success($status);
    }

    /**
     * AJAX: Test deployment configuration
     */
    public function ajax_test_deployment() {
        check_ajax_referer('vssv_ajax_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }

        $platform = $_POST['platform'] ?? '';
        $results = [];

        if ($platform && $this->is_platform_configured($platform)) {
            // Test webhook or API connection
            $config = $this->settings['platforms'][$platform];

            if (!empty($config['webhook_url'])) {
                $response = wp_remote_head($config['webhook_url']);
                $results['webhook'] = !is_wp_error($response);
            }

            // Platform-specific tests
            if ($platform === 'cloudflare' && !empty($config['api_token'])) {
                $test = wp_remote_get('https://api.cloudflare.com/client/v4/user', [
                    'headers' => ['Authorization' => 'Bearer ' . $config['api_token']]
                ]);
                $results['api'] = !is_wp_error($test) && wp_remote_retrieve_response_code($test) === 200;
            }
        }

        wp_send_json_success($results);
    }

    /**
     * AJAX: Rollback deployment
     */
    public function ajax_rollback() {
        check_ajax_referer('vssv_ajax_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }

        $platform = $_POST['platform'] ?? $this->settings['primary_platform'];
        $deployment_id = $_POST['deployment_id'] ?? '';

        if (empty($deployment_id)) {
            wp_send_json_error('No deployment ID provided');
        }

        if (!$this->settings['rollback_enabled']) {
            wp_send_json_error('Rollback is disabled');
        }

        $result = $this->rollback_deployment($platform, $deployment_id);
        wp_send_json($result);
    }

    /**
     * REST: Rollback deployment
     */
    public function rest_rollback($request) {
        $platform = $request->get_param('platform');
        $deployment_id = $request->get_param('deployment_id') ?? '';

        if (empty($platform) || empty($deployment_id)) {
            return new WP_REST_Response(['error' => 'Missing required parameters'], 400);
        }

        if (!$this->settings['rollback_enabled']) {
            return new WP_REST_Response(['error' => 'Rollback is disabled'], 403);
        }

        $result = $this->rollback_deployment($platform, $deployment_id);
        return new WP_REST_Response($result);
    }

    /**
     * Rollback deployment to previous version
     */
    private function rollback_deployment($platform, $deployment_id) {
        if (!$this->is_platform_configured($platform)) {
            return [
                'success' => false,
                'error' => 'Platform not configured'
            ];
        }

        $config = $this->settings['platforms'][$platform];

        // Rollback via webhook
        if (!empty($config['webhook_url'])) {
            return $this->rollback_via_webhook($platform, $deployment_id);
        }

        // Rollback via API (Cloudflare)
        if ($platform === 'cloudflare' && !empty($config['api_token'])) {
            return $this->rollback_via_api($platform, $deployment_id, $config);
        }

        return [
            'success' => false,
            'error' => 'Rollback method not configured for this platform'
        ];
    }

    /**
     * Rollback via webhook
     */
    private function rollback_via_webhook($platform, $deployment_id) {
        $config = $this->settings['platforms'][$platform];

        $response = wp_remote_post($config['webhook_url'], [
            'timeout' => 10,
            'body' => [
                'action' => 'rollback',
                'platform' => $platform,
                'deployment_id' => $deployment_id,
                'source' => 'vs_site_vector'
            ]
        ]);

        if (is_wp_error($response)) {
            return [
                'success' => false,
                'error' => $response->get_error_message()
            ];
        }

        $code = wp_remote_retrieve_response_code($response);
        if ($code >= 200 && $code < 300) {
            return [
                'success' => true,
                'message' => 'Deployment rolled back successfully'
            ];
        }

        return [
            'success' => false,
            'error' => 'Rollback failed with status code ' . $code
        ];
    }

    /**
     * Rollback via API (Cloudflare)
     */
    private function rollback_via_api($platform, $deployment_id, $config) {
        if ($platform !== 'cloudflare') {
            return [
                'success' => false,
                'error' => 'API rollback only supported for Cloudflare'
            ];
        }

        // Note: This is a placeholder. Cloudflare Pages API doesn't have a direct rollback endpoint.
        // Rollback would be handled via webhook or through the platform's web interface.
        return [
            'success' => false,
            'error' => 'API-based rollback not available. Use webhook or platform dashboard.'
        ];
    }

    /**
     * Get recommendations
     */
    public function get_recommendations() {
        $recommendations = [];

        if (!$this->settings['enabled']) {
            $recommendations[] = [
                'type' => 'info',
                'message' => 'Deployment module is disabled',
                'action' => 'Enable deployment to push static sites to CDN'
            ];
        }

        $configured_platforms = array_filter($this->settings['platforms'], function($p) {
            return $p['enabled'];
        });

        if (empty($configured_platforms)) {
            $recommendations[] = [
                'type' => 'warning',
                'message' => 'No deployment platforms configured',
                'action' => 'Configure at least one deployment platform'
            ];
        }

        return $recommendations;
    }

    /**
     * Run optimization
     */
    public function optimize() {
        $results = [
            'deployment_enabled' => $this->settings['enabled'],
            'platforms_configured' => count(array_filter($this->settings['platforms'], function($p) {
                return $p['enabled'];
            })),
            'auto_deploy' => $this->settings['auto_deploy']
        ];

        return $results;
    }
}