<?php
/**
 * VS Site Vector Diagnostics Module
 *
 * Provides comprehensive diagnostic testing and reporting
 * for VS Site Vector performance optimization.
 *
 * @package VS_Site_Vector
 * @since 1.2.1
 */

if (!defined('ABSPATH')) {
    exit;
}

class VSSV_Diagnostics {

    /**
     * Instance of this class
     */
    private static $instance = null;

    /**
     * Get singleton instance
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        $this->init_hooks();
    }

    /**
     * Initialize hooks
     */
    private function init_hooks() {
        // Enqueue diagnostic script when ?vssv_debug=1
        add_action('wp_enqueue_scripts', [$this, 'maybe_enqueue_diagnostics']);
        add_action('admin_enqueue_scripts', [$this, 'maybe_enqueue_diagnostics']);

        // Register AJAX handlers
        add_action('wp_ajax_vssv_diagnostic_check', [$this, 'ajax_diagnostic_check']);
        add_action('wp_ajax_vssv_get_cache_status', [$this, 'ajax_get_cache_status']);
        add_action('wp_ajax_vssv_get_deploy_status', [$this, 'ajax_get_deploy_status']);
        add_action('wp_ajax_vssv_run_diagnostic_test', [$this, 'ajax_run_diagnostic_test']);
    }

    /**
     * Conditionally enqueue diagnostic script
     */
    public function maybe_enqueue_diagnostics() {
        // Only load when explicitly requested
        if (!isset($_GET['vssv_debug']) || $_GET['vssv_debug'] !== '1') {
            return;
        }

        // Enqueue jQuery dependency
        wp_enqueue_script('jquery');

        // Enqueue diagnostics script
        wp_enqueue_script(
            'vssv-diagnostics',
            VSSV_PLUGIN_URL . 'assets/js/vssv-diagnostics.js',
            ['jquery'],
            VSSV_VERSION,
            true
        );

        // Prepare localization data
        $diag_data = [
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'version' => VSSV_VERSION,
            'isAdmin' => current_user_can('manage_options')
        ];

        // Only include nonce for admins
        if (current_user_can('manage_options')) {
            $diag_data['nonce'] = wp_create_nonce('vssv-diagnostic-nonce');
        }

        wp_localize_script('vssv-diagnostics', 'vssvDiag', $diag_data);

        // Also provide config data for frontend checks
        $config_data = [
            'version' => VSSV_VERSION,
            'cacheEnabled' => (bool) get_option('vssv_cache_enabled', true),
            'optimizerActive' => true,
            'modules' => $this->get_active_modules_status()
        ];

        wp_localize_script('vssv-diagnostics', 'vssvConfig', $config_data);
    }

    /**
     * Get active modules status
     */
    private function get_active_modules_status() {
        $active_modules = get_option('vssv_active_modules', ['cache', 'static', 'deploy', 'api']);

        return [
            'cache' => in_array('cache', $active_modules),
            'static' => in_array('static', $active_modules),
            'deploy' => in_array('deploy', $active_modules),
            'api' => in_array('api', $active_modules),
            'optimizer' => true // Always active
        ];
    }

    /**
     * AJAX: Run comprehensive diagnostic check
     */
    public function ajax_diagnostic_check() {
        // Verify nonce and capabilities
        if (!check_ajax_referer('vssv-diagnostic-nonce', 'nonce', false)) {
            wp_send_json_error(['message' => 'Invalid security token']);
            return;
        }

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Insufficient permissions']);
            return;
        }

        global $wpdb;

        $diagnostics = [];

        // PHP Version
        $diagnostics['php_version'] = PHP_VERSION;

        // WordPress Version
        $diagnostics['wp_version'] = get_bloginfo('version');

        // Plugin Version
        $diagnostics['plugin_version'] = VSSV_VERSION;

        // Database Tables Check
        $required_tables = [
            $wpdb->prefix . 'vssv_metrics',
            $wpdb->prefix . 'vssv_optimization_log'
        ];

        $missing_tables = [];
        foreach ($required_tables as $table) {
            $exists = $wpdb->get_var($wpdb->prepare(
                "SHOW TABLES LIKE %s",
                $table
            ));
            if (!$exists) {
                $missing_tables[] = $table;
            }
        }

        $diagnostics['tables_exist'] = empty($missing_tables);
        $diagnostics['missing_tables'] = $missing_tables;

        // Cache Directory Check
        $diagnostics['cache_writable'] = is_writable(VSSV_CACHE_DIR);
        $diagnostics['cache_dir'] = VSSV_CACHE_DIR;

        // Cache Directory Size
        $diagnostics['cache_size'] = $this->get_directory_size(VSSV_CACHE_DIR);

        // PHP Extensions
        $required_extensions = ['mbstring', 'dom', 'xml', 'json'];
        $recommended_extensions = ['imagick', 'redis', 'apcu', 'opcache'];

        $extensions = [
            'required' => [],
            'recommended' => []
        ];

        foreach ($required_extensions as $ext) {
            $extensions['required'][$ext] = extension_loaded($ext);
        }

        foreach ($recommended_extensions as $ext) {
            $extensions['recommended'][$ext] = extension_loaded($ext);
        }

        $diagnostics['extensions'] = $extensions;

        // Active Modules
        $diagnostics['active_modules'] = get_option('vssv_active_modules', ['cache', 'static', 'deploy', 'api']);

        // Performance Score
        $diagnostics['performance_score'] = (int) get_option('vssv_performance_score', 0);
        $diagnostics['performance_factors'] = get_option('vssv_performance_factors', []);

        // Cache Statistics
        $diagnostics['cache_stats'] = $this->get_cache_statistics();

        // Recent Optimizations (last 7 days)
        $recent_optimizations = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$wpdb->prefix}vssv_optimization_log WHERE timestamp > %s",
            gmdate('Y-m-d H:i:s', strtotime('-7 days'))
        ));
        $diagnostics['recent_optimizations'] = (int) $recent_optimizations;

        // Memory Limit
        $diagnostics['memory_limit'] = ini_get('memory_limit');

        // Max Execution Time
        $diagnostics['max_execution_time'] = ini_get('max_execution_time');

        // Server Software
        $diagnostics['server_software'] = isset($_SERVER['SERVER_SOFTWARE']) ?
            sanitize_text_field(wp_unslash($_SERVER['SERVER_SOFTWARE'])) : 'Unknown';

        // Emergency/Safe Mode Status
        $diagnostics['emergency_disabled'] = VSSV_Config::is_emergency_disabled();
        $diagnostics['safe_mode'] = VSSV_Config::is_safe_mode();

        // Object Cache Status
        $diagnostics['object_cache'] = [
            'enabled' => wp_using_ext_object_cache(),
            'type' => $this->detect_object_cache_type()
        ];

        // OPcache Status
        $diagnostics['opcache'] = $this->get_opcache_status();

        wp_send_json_success($diagnostics);
    }

    /**
     * AJAX: Get cache status
     */
    public function ajax_get_cache_status() {
        // Verify nonce and capabilities
        if (!check_ajax_referer('vssv-diagnostic-nonce', 'nonce', false)) {
            wp_send_json_error(['message' => 'Invalid security token']);
            return;
        }

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Insufficient permissions']);
            return;
        }

        $cache_status = [];

        // Page Cache
        $page_cache_enabled = get_option('vssv_cache_enabled', true);
        $page_cache_dir = VSSV_CACHE_DIR . 'html/';

        $page_cache_entries = 0;
        $page_cache_size = '0 KB';

        if (is_dir($page_cache_dir)) {
            $page_cache_entries = count(glob($page_cache_dir . '*.html'));
            $page_cache_size = $this->get_directory_size($page_cache_dir);
        }

        $cache_status['page_cache'] = [
            'enabled' => (bool) $page_cache_enabled,
            'entries' => $page_cache_entries,
            'size' => $page_cache_size,
            'dir' => $page_cache_dir
        ];

        // Object Cache
        $cache_status['object_cache'] = [
            'enabled' => wp_using_ext_object_cache(),
            'type' => $this->detect_object_cache_type(),
            'persistent' => wp_using_ext_object_cache()
        ];

        // Fragment Cache
        $fragment_cache_dir = VSSV_CACHE_DIR . 'fragments/';
        $fragment_entries = 0;

        if (is_dir($fragment_cache_dir)) {
            $fragment_entries = count(glob($fragment_cache_dir . '*'));
        }

        $cache_status['fragment_cache'] = [
            'enabled' => (bool) get_option('vssv_fragment_cache_enabled', false),
            'entries' => $fragment_entries
        ];

        // Critical CSS Cache
        $critical_css_dir = VSSV_CACHE_DIR . 'critical/';
        $critical_entries = 0;

        if (is_dir($critical_css_dir)) {
            $critical_entries = count(glob($critical_css_dir . '*.css'));
        }

        $cache_status['critical_css'] = [
            'enabled' => (bool) get_option('vssv_critical_css_enabled', false),
            'entries' => $critical_entries
        ];

        // Cache Hit/Miss Stats
        $cache_status['statistics'] = $this->get_cache_statistics();

        wp_send_json_success($cache_status);
    }

    /**
     * AJAX: Get CDN deployment status
     */
    public function ajax_get_deploy_status() {
        // Verify nonce and capabilities
        if (!check_ajax_referer('vssv-diagnostic-nonce', 'nonce', false)) {
            wp_send_json_error(['message' => 'Invalid security token']);
            return;
        }

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Insufficient permissions']);
            return;
        }

        $deploy_status = [];

        // Check if deploy module is active
        $active_modules = get_option('vssv_active_modules', []);
        $deploy_status['enabled'] = in_array('deploy', $active_modules);

        if ($deploy_status['enabled']) {
            // CDN Configuration
            $cdn_provider = get_option('vssv_cdn_provider', '');
            $deploy_status['configured'] = !empty($cdn_provider);
            $deploy_status['provider'] = $cdn_provider ?: 'None';

            // Connection status (check if credentials are set)
            $cdn_key = get_option('vssv_cdn_api_key', '');
            $deploy_status['connected'] = !empty($cdn_key);

            // Last deployment
            $last_deploy = get_option('vssv_last_deployment', null);
            $deploy_status['last_deploy'] = $last_deploy ?
                date_i18n(get_option('date_format') . ' ' . get_option('time_format'), $last_deploy) :
                'Never';

            // Deployment stats
            $deploy_status['total_deployments'] = (int) get_option('vssv_deployment_count', 0);
        }

        wp_send_json_success($deploy_status);
    }

    /**
     * AJAX: Run individual diagnostic test
     */
    public function ajax_run_diagnostic_test() {
        // Verify nonce and capabilities
        if (!check_ajax_referer('vssv-diagnostic-nonce', 'nonce', false)) {
            wp_send_json_error(['message' => 'Invalid security token']);
            return;
        }

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Insufficient permissions']);
            return;
        }

        $test_name = isset($_POST['test']) ? sanitize_text_field(wp_unslash($_POST['test'])) : '';

        switch ($test_name) {
            case 'cache_write':
                $result = $this->test_cache_write();
                break;

            case 'database':
                $result = $this->test_database_connection();
                break;

            case 'memory':
                $result = $this->test_memory_availability();
                break;

            case 'opcache':
                $result = $this->get_opcache_status();
                break;

            default:
                wp_send_json_error(['message' => 'Unknown test: ' . $test_name]);
                return;
        }

        wp_send_json_success($result);
    }

    /**
     * Get cache statistics
     */
    private function get_cache_statistics() {
        $stats = [
            'hits' => (int) get_transient('vssv_cache_hits') ?: 0,
            'misses' => (int) get_transient('vssv_cache_misses') ?: 0,
            'writes' => (int) get_transient('vssv_cache_writes') ?: 0,
            'deletes' => (int) get_transient('vssv_cache_deletes') ?: 0
        ];

        $stats['total'] = $stats['hits'] + $stats['misses'];
        $stats['hit_rate'] = $stats['total'] > 0 ?
            round(($stats['hits'] / $stats['total']) * 100, 2) : 0;

        return $stats;
    }

    /**
     * Get directory size in human readable format
     */
    private function get_directory_size($dir) {
        if (!is_dir($dir)) {
            return '0 KB';
        }

        $size = 0;
        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($dir, RecursiveDirectoryIterator::SKIP_DOTS)
        );

        foreach ($iterator as $file) {
            if ($file->isFile()) {
                $size += $file->getSize();
            }
        }

        return $this->format_bytes($size);
    }

    /**
     * Format bytes to human readable
     */
    private function format_bytes($bytes) {
        $units = ['B', 'KB', 'MB', 'GB'];
        $i = 0;

        while ($bytes >= 1024 && $i < count($units) - 1) {
            $bytes /= 1024;
            $i++;
        }

        return round($bytes, 2) . ' ' . $units[$i];
    }

    /**
     * Detect object cache type
     */
    private function detect_object_cache_type() {
        if (!wp_using_ext_object_cache()) {
            return 'None (Default)';
        }

        if (extension_loaded('redis') && class_exists('Redis')) {
            return 'Redis';
        }

        if (extension_loaded('memcached') && class_exists('Memcached')) {
            return 'Memcached';
        }

        if (extension_loaded('apcu') && function_exists('apcu_store')) {
            return 'APCu';
        }

        return 'External (Unknown type)';
    }

    /**
     * Get OPcache status
     */
    private function get_opcache_status() {
        if (!function_exists('opcache_get_status')) {
            return [
                'enabled' => false,
                'message' => 'OPcache not available'
            ];
        }

        $status = @opcache_get_status(false);

        if (!$status) {
            return [
                'enabled' => false,
                'message' => 'OPcache disabled or restricted'
            ];
        }

        return [
            'enabled' => $status['opcache_enabled'],
            'memory_used' => isset($status['memory_usage']) ?
                $this->format_bytes($status['memory_usage']['used_memory']) : 'N/A',
            'memory_free' => isset($status['memory_usage']) ?
                $this->format_bytes($status['memory_usage']['free_memory']) : 'N/A',
            'hit_rate' => isset($status['opcache_statistics']) ?
                round($status['opcache_statistics']['opcache_hit_rate'], 2) . '%' : 'N/A',
            'cached_scripts' => isset($status['opcache_statistics']) ?
                $status['opcache_statistics']['num_cached_scripts'] : 0
        ];
    }

    /**
     * Test cache write functionality
     */
    private function test_cache_write() {
        $test_file = VSSV_CACHE_DIR . 'test_' . time() . '.txt';
        $test_content = 'VSSV diagnostic test - ' . date('Y-m-d H:i:s');

        // Try to write
        $write_result = @file_put_contents($test_file, $test_content);

        if ($write_result === false) {
            return [
                'success' => false,
                'message' => 'Cannot write to cache directory',
                'path' => VSSV_CACHE_DIR
            ];
        }

        // Verify content
        $read_content = @file_get_contents($test_file);
        $content_match = ($read_content === $test_content);

        // Cleanup
        @unlink($test_file);

        return [
            'success' => $content_match,
            'message' => $content_match ?
                'Cache write/read successful' :
                'Cache write succeeded but read verification failed',
            'path' => VSSV_CACHE_DIR
        ];
    }

    /**
     * Test database connection and performance
     */
    private function test_database_connection() {
        global $wpdb;

        $start = microtime(true);

        // Simple query test
        $result = $wpdb->get_var("SELECT 1");

        $query_time = round((microtime(true) - $start) * 1000, 2);

        if ($result !== '1') {
            return [
                'success' => false,
                'message' => 'Database query failed',
                'query_time' => $query_time . 'ms'
            ];
        }

        // Check tables
        $tables = $wpdb->get_results("SHOW TABLES LIKE '{$wpdb->prefix}vssv_%'");

        return [
            'success' => true,
            'message' => 'Database connection successful',
            'query_time' => $query_time . 'ms',
            'vssv_tables' => count($tables),
            'db_version' => $wpdb->db_version()
        ];
    }

    /**
     * Test available memory
     */
    private function test_memory_availability() {
        $memory_limit = ini_get('memory_limit');
        $memory_limit_bytes = $this->convert_to_bytes($memory_limit);

        $current_usage = memory_get_usage(true);
        $peak_usage = memory_get_peak_usage(true);

        $available = $memory_limit_bytes - $current_usage;
        $usage_percent = round(($current_usage / $memory_limit_bytes) * 100, 2);

        return [
            'success' => $usage_percent < 80,
            'message' => $usage_percent < 80 ?
                'Adequate memory available' :
                'Memory usage is high',
            'limit' => $memory_limit,
            'current' => $this->format_bytes($current_usage),
            'peak' => $this->format_bytes($peak_usage),
            'available' => $this->format_bytes($available),
            'usage_percent' => $usage_percent . '%'
        ];
    }

    /**
     * Convert memory string to bytes
     */
    private function convert_to_bytes($value) {
        $value = trim($value);
        $unit = strtolower(substr($value, -1));
        $value = (int) $value;

        switch ($unit) {
            case 'g':
                $value *= 1024 * 1024 * 1024;
                break;
            case 'm':
                $value *= 1024 * 1024;
                break;
            case 'k':
                $value *= 1024;
                break;
        }

        return $value;
    }

    /**
     * Generate diagnostic report
     */
    public function generate_report() {
        $report = [];

        $report['generated'] = current_time('mysql');
        $report['plugin_version'] = VSSV_VERSION;
        $report['php_version'] = PHP_VERSION;
        $report['wp_version'] = get_bloginfo('version');

        // Server Info
        $report['server'] = [
            'software' => isset($_SERVER['SERVER_SOFTWARE']) ?
                sanitize_text_field(wp_unslash($_SERVER['SERVER_SOFTWARE'])) : 'Unknown',
            'memory_limit' => ini_get('memory_limit'),
            'max_execution_time' => ini_get('max_execution_time'),
            'upload_max_filesize' => ini_get('upload_max_filesize')
        ];

        // Plugin Status
        $report['modules'] = $this->get_active_modules_status();
        $report['cache_stats'] = $this->get_cache_statistics();
        $report['performance_score'] = get_option('vssv_performance_score', 0);

        // Extensions
        $report['extensions'] = [
            'opcache' => extension_loaded('opcache'),
            'redis' => extension_loaded('redis'),
            'memcached' => extension_loaded('memcached'),
            'imagick' => extension_loaded('imagick'),
            'gd' => extension_loaded('gd')
        ];

        return $report;
    }
}

// Initialize diagnostics
add_action('plugins_loaded', function() {
    VSSV_Diagnostics::get_instance();
}, 15);
