<?php
/**
 * Core Optimizer Class
 *
 * Handles HTML, CSS, JS optimization and performance enhancements
 *
 * @package VS_Site_Vector
 * @since 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class VSSV_Optimizer {

    /**
     * Optimization settings
     */
    private $settings = [];

    /**
     * Performance metrics
     */
    private $metrics = [];

    /**
     * Constructor
     */
    public function __construct() {
        $this->load_settings();
        $this->init_hooks();
    }

    /**
     * Load optimization settings
     */
    private function load_settings() {
        $this->settings = [
            'minify_html' => get_option('vssv_minify_html', true),
            'remove_comments' => get_option('vssv_remove_html_comments', true),
            'remove_whitespace' => get_option('vssv_remove_whitespace', true),
            'defer_js' => get_option('vssv_defer_js', false), // Default OFF - can break functionality
            'async_js' => get_option('vssv_async_js', []),
            'critical_css' => get_option('vssv_critical_css', false), // Default OFF - requires setup
            'inline_small_css' => get_option('vssv_inline_small_css', true),
            'inline_small_js' => get_option('vssv_inline_small_js', false), // Default OFF - can break dependencies
            'remove_query_strings' => get_option('vssv_remove_query_strings', true),
            'dns_prefetch' => get_option('vssv_dns_prefetch', true),
            'preconnect' => get_option('vssv_preconnect_domains', []),
            'resource_hints' => get_option('vssv_resource_hints', true),
            'lazy_load' => get_option('vssv_lazy_load', true),
            'disable_emojis' => get_option('vssv_disable_emojis', true),
            'disable_embeds' => get_option('vssv_disable_embeds', true),
            'remove_shortlink' => get_option('vssv_remove_shortlink', true),
            'remove_wlwmanifest' => get_option('vssv_remove_wlwmanifest', true),
            'remove_rest_api_links' => get_option('vssv_remove_rest_api_links', false),
            'remove_feed_links' => get_option('vssv_remove_feed_links', false)
        ];
    }

    /**
     * Initialize hooks
     */
    private function init_hooks() {
        // Resource hints
        if ($this->settings['dns_prefetch']) {
            add_action('wp_head', [$this, 'add_dns_prefetch'], 1);
        }

        if ($this->settings['resource_hints']) {
            add_filter('wp_resource_hints', [$this, 'add_resource_hints'], 10, 2);
        }

        // Remove unnecessary items
        if ($this->settings['disable_emojis']) {
            $this->disable_emojis();
        }

        if ($this->settings['disable_embeds']) {
            $this->disable_embeds();
        }

        if ($this->settings['remove_shortlink']) {
            remove_action('wp_head', 'wp_shortlink_wp_head');
        }

        if ($this->settings['remove_wlwmanifest']) {
            remove_action('wp_head', 'wlwmanifest_link');
        }

        if ($this->settings['remove_rest_api_links']) {
            remove_action('wp_head', 'rest_output_link_wp_head');
        }

        if ($this->settings['remove_feed_links']) {
            remove_action('wp_head', 'feed_links', 2);
            remove_action('wp_head', 'feed_links_extra', 3);
        }

        // Query strings
        if ($this->settings['remove_query_strings']) {
            add_filter('script_loader_src', [$this, 'remove_query_strings'], 15, 1);
            add_filter('style_loader_src', [$this, 'remove_query_strings'], 15, 1);
        }

        // JavaScript optimization
        if ($this->settings['defer_js']) {
            add_filter('script_loader_tag', [$this, 'defer_js_scripts'], 10, 3);
        }

        // CSS optimization
        add_filter('style_loader_tag', [$this, 'optimize_css_loading'], 10, 4);

        // Preload key resources
        add_action('wp_head', [$this, 'preload_key_resources'], 2);

        // Remove unnecessary meta tags
        remove_action('wp_head', 'wp_generator');
        remove_action('wp_head', 'rsd_link');
        remove_action('wp_head', 'wp_oembed_add_discovery_links');
    }

    /**
     * Optimize HTML output
     */
    public function optimize_html($html) {
        if (empty($html)) {
            return $html;
        }

        $start_time = microtime(true);

        // Remove HTML comments (preserve IE conditionals)
        if ($this->settings['remove_comments']) {
            $html = preg_replace('/<!--(?!\s*(?:\[if [^\]]+\]|<!|>))(?:(?!-->).)*-->/s', '', $html);
        }

        // Minify HTML
        if ($this->settings['minify_html']) {
            $html = $this->minify_html($html);
        }

        // Inline critical CSS
        if ($this->settings['critical_css']) {
            $html = $this->inject_critical_css($html);
        }

        // Inline small CSS files
        if ($this->settings['inline_small_css']) {
            $html = $this->inline_small_css($html);
        }

        // Inline small JS files
        if ($this->settings['inline_small_js']) {
            $html = $this->inline_small_js($html);
        }

        // Add lazy loading
        if ($this->settings['lazy_load']) {
            $html = $this->add_lazy_loading($html);
        }

        // Optimize images
        $html = $this->optimize_images_in_html($html);

        // Add performance metrics
        $optimization_time = microtime(true) - $start_time;
        $this->metrics['html_optimization_time'] = $optimization_time;

        // Inject performance comment (for debugging)
        if (defined('WP_DEBUG') && WP_DEBUG) {
            $html = str_replace(
                '</html>',
                sprintf(
                    "\n<!-- VS Site Vector: HTML optimized in %.4fs -->\n</html>",
                    $optimization_time
                ),
                $html
            );
        }

        return $html;
    }

    /**
     * Minify HTML
     */
    private function minify_html($html) {
        // Protect pre, textarea, script, and style tags from minification
        // These tags contain code that relies on whitespace and formatting
        $protected_content = [];

        $html = preg_replace_callback(
            '/<(pre|textarea|script|style)(?:\s[^>]*)?>.*?<\/\1>/is',
            function($matches) use (&$protected_content) {
                $placeholder = '___PROTECTED_' . md5($matches[0]) . '___';
                $protected_content[$placeholder] = $matches[0];
                return $placeholder;
            },
            $html
        );

        // Remove whitespace
        if ($this->settings['remove_whitespace']) {
            // Remove whitespace between tags
            $html = preg_replace('/>\s+</', '><', $html);

            // Remove multiple whitespace
            $html = preg_replace('/\s+/', ' ', $html);

            // Remove whitespace around block elements
            $blocks = 'address|article|aside|blockquote|canvas|dd|div|dl|dt|fieldset|figcaption|figure|footer|form|h[1-6]|header|hr|li|main|nav|noscript|ol|output|p|pre|section|table|tfoot|ul|video';
            $html = preg_replace('/\s*(<(?:' . $blocks . ')(?:\s[^>]*)?>)\s*/i', '$1', $html);
            $html = preg_replace('/\s*(<\/(?:' . $blocks . ')>)\s*/i', '$1', $html);
        }

        // Restore protected tags (script, style, pre, textarea)
        foreach ($protected_content as $placeholder => $original) {
            $html = str_replace($placeholder, $original, $html);
        }

        return $html;
    }

    /**
     * Inject critical CSS
     */
    private function inject_critical_css($html) {
        $critical_css = $this->get_critical_css();

        if (!empty($critical_css)) {
            // Add critical CSS inline
            $critical_style = '<style id="vssv-critical-css">' . $critical_css . '</style>';
            $html = str_replace('</head>', $critical_style . '</head>', $html);

            // Make other CSS async
            $html = preg_replace_callback(
                '/<link([^>]*?)href=["\']([^"\']+\.css[^"\']*)["\']([^>]*?)>/i',
                function($matches) {
                    // Skip if already has media attribute for print
                    if (strpos($matches[0], 'media="print"') !== false) {
                        return $matches[0];
                    }

                    // Make CSS load asynchronously
                    return sprintf(
                        '<link%s href="%s"%s media="print" onload="this.onload=null;this.media=\'all\'">
                        <noscript><link%s href="%s"%s></noscript>',
                        $matches[1], $matches[2], $matches[3],
                        $matches[1], $matches[2], $matches[3]
                    );
                },
                $html
            );
        }

        return $html;
    }

    /**
     * Get critical CSS for current page
     */
    private function get_critical_css() {
        // Check cache first
        $cache_key = 'vssv_critical_' . md5($_SERVER['REQUEST_URI']);
        $cached = get_transient($cache_key);

        if ($cached !== false) {
            return $cached;
        }

        // Generate critical CSS
        $critical_css = '';

        // Get page-specific critical CSS
        if (is_front_page()) {
            $critical_css = get_option('vssv_critical_css_home', '');
        } elseif (is_singular()) {
            global $post;
            $critical_css = get_post_meta($post->ID, '_vssv_critical_css', true);
        }

        // Fallback to global critical CSS
        if (empty($critical_css)) {
            $critical_css = get_option('vssv_critical_css_global', '');
        }

        // Minify critical CSS
        if (!empty($critical_css)) {
            $critical_css = $this->minify_css($critical_css);
        }

        // Cache for 24 hours
        set_transient($cache_key, $critical_css, DAY_IN_SECONDS);

        return $critical_css;
    }

    /**
     * Inline small CSS files
     */
    private function inline_small_css($html) {
        $max_size = 5000; // 5KB threshold

        $html = preg_replace_callback(
            '/<link([^>]*?)href=["\']([^"\']+\.css[^"\']*)["\']([^>]*?)>/i',
            function($matches) use ($max_size) {
                $url = $matches[2];

                // Skip external CSS
                if (strpos($url, home_url()) === false && strpos($url, '/') !== 0) {
                    return $matches[0];
                }

                // Get file path
                $path = $this->url_to_path($url);
                if (!$path || !file_exists($path)) {
                    return $matches[0];
                }

                // Check file size
                if (filesize($path) > $max_size) {
                    return $matches[0];
                }

                // Read and minify CSS
                $css = file_get_contents($path);
                $css = $this->minify_css($css);

                // Inline the CSS
                return '<style' . $matches[1] . $matches[3] . '>' . $css . '</style>';
            },
            $html
        );

        return $html;
    }

    /**
     * Inline small JavaScript files
     */
    private function inline_small_js($html) {
        $max_size = 3000; // 3KB threshold

        $html = preg_replace_callback(
            '/<script([^>]*?)src=["\']([^"\']+\.js[^"\']*)["\']([^>]*?)><\/script>/i',
            function($matches) use ($max_size) {
                $url = $matches[2];

                // Skip external JS
                if (strpos($url, home_url()) === false && strpos($url, '/') !== 0) {
                    return $matches[0];
                }

                // Get file path
                $path = $this->url_to_path($url);
                if (!$path || !file_exists($path)) {
                    return $matches[0];
                }

                // Check file size
                if (filesize($path) > $max_size) {
                    return $matches[0];
                }

                // Read and minify JS
                $js = file_get_contents($path);
                $js = $this->minify_js($js);

                // Inline the JS
                return '<script' . $matches[1] . $matches[3] . '>' . $js . '</script>';
            },
            $html
        );

        return $html;
    }

    /**
     * Add lazy loading to images and iframes
     */
    private function add_lazy_loading($html) {
        // Add lazy loading to images
        $html = preg_replace_callback(
            '/<img([^>]*?)src=["\']([^"\']+)["\']([^>]*?)>/i',
            function($matches) {
                // Skip if already has loading attribute
                if (strpos($matches[0], 'loading=') !== false) {
                    return $matches[0];
                }

                // Skip logo and above-the-fold images
                if (strpos($matches[0], 'logo') !== false ||
                    strpos($matches[0], 'hero') !== false ||
                    strpos($matches[0], 'no-lazy') !== false) {
                    return $matches[0];
                }

                // Add loading="lazy"
                return '<img' . $matches[1] . 'src="' . $matches[2] . '"' . $matches[3] . ' loading="lazy">';
            },
            $html
        );

        // Add lazy loading to iframes
        $html = preg_replace_callback(
            '/<iframe([^>]*?)src=["\']([^"\']+)["\']([^>]*?)>/i',
            function($matches) {
                // Skip if already has loading attribute
                if (strpos($matches[0], 'loading=') !== false) {
                    return $matches[0];
                }

                // Add loading="lazy"
                return '<iframe' . $matches[1] . 'src="' . $matches[2] . '"' . $matches[3] . ' loading="lazy">';
            },
            $html
        );

        return $html;
    }

    /**
     * Optimize images in HTML
     */
    private function optimize_images_in_html($html) {
        $html = preg_replace_callback(
            '/<img([^>]*?)>/i',
            function($matches) {
                $img_tag = $matches[0];

                // Add width and height if missing (prevents layout shift)
                if (strpos($img_tag, 'width=') === false || strpos($img_tag, 'height=') === false) {
                    // Try to get dimensions from URL
                    preg_match('/src=["\']([^"\']+)["\']/i', $img_tag, $src_match);
                    if (!empty($src_match[1])) {
                        $dimensions = $this->get_image_dimensions($src_match[1]);
                        if ($dimensions) {
                            $img_tag = str_replace('<img', '<img width="' . $dimensions['width'] . '" height="' . $dimensions['height'] . '"', $img_tag);
                        }
                    }
                }

                // Add decoding="async" for better performance
                if (strpos($img_tag, 'decoding=') === false) {
                    $img_tag = str_replace('<img', '<img decoding="async"', $img_tag);
                }

                return $img_tag;
            },
            $html
        );

        return $html;
    }

    /**
     * Defer JavaScript loading
     */
    public function defer_js_scripts($tag, $handle, $src) {
        // Skip admin scripts
        if (is_admin()) {
            return $tag;
        }

        // Skip jQuery and other critical scripts
        $skip_handles = ['jquery', 'jquery-core', 'jquery-migrate'];
        if (in_array($handle, $skip_handles)) {
            return $tag;
        }

        // Skip inline scripts
        if (empty($src)) {
            return $tag;
        }

        // Skip if already has defer or async
        if (strpos($tag, 'defer') !== false || strpos($tag, 'async') !== false) {
            return $tag;
        }

        // Add defer attribute
        return str_replace(' src', ' defer src', $tag);
    }

    /**
     * Optimize CSS loading
     */
    public function optimize_css_loading($html, $handle, $href, $media) {
        // Skip admin styles
        if (is_admin()) {
            return $html;
        }

        // Skip print styles
        if ($media === 'print') {
            return $html;
        }

        // Check if this CSS should be loaded asynchronously
        $async_handles = get_option('vssv_async_css_handles', []);
        if (in_array($handle, $async_handles)) {
            // Load CSS asynchronously
            return sprintf(
                '<link rel="stylesheet" href="%s" media="print" onload="this.onload=null;this.media=\'%s\'">
                <noscript><link rel="stylesheet" href="%s" media="%s"></noscript>',
                $href, $media, $href, $media
            );
        }

        return $html;
    }

    /**
     * Add DNS prefetch
     */
    public function add_dns_prefetch() {
        $domains = [
            '//fonts.googleapis.com',
            '//fonts.gstatic.com',
            '//cdnjs.cloudflare.com',
            '//www.google-analytics.com',
            '//www.googletagmanager.com'
        ];

        // Add custom domains
        $custom_domains = get_option('vssv_dns_prefetch_domains', []);
        $domains = array_merge($domains, $custom_domains);

        foreach ($domains as $domain) {
            echo '<link rel="dns-prefetch" href="' . esc_url($domain) . '">' . "\n";
        }
    }

    /**
     * Add resource hints
     */
    public function add_resource_hints($hints, $relation_type) {
        if ($relation_type === 'preconnect') {
            $hints[] = 'https://fonts.googleapis.com';
            $hints[] = 'https://fonts.gstatic.com';

            // Add custom preconnect domains
            $custom = $this->settings['preconnect'];
            if (is_array($custom)) {
                $hints = array_merge($hints, $custom);
            }
        }

        return $hints;
    }

    /**
     * Preload key resources
     */
    public function preload_key_resources() {
        // Preload fonts
        $fonts = get_option('vssv_preload_fonts', []);
        foreach ($fonts as $font) {
            printf(
                '<link rel="preload" href="%s" as="font" type="font/%s" crossorigin>',
                esc_url($font['url']),
                esc_attr($font['type'])
            );
        }

        // Preload critical images
        if (is_front_page()) {
            $hero_image = get_option('vssv_hero_image');
            if ($hero_image) {
                printf(
                    '<link rel="preload" as="image" href="%s">',
                    esc_url($hero_image)
                );
            }
        }

        // Preload critical CSS
        $critical_css_url = get_option('vssv_critical_css_url');
        if ($critical_css_url) {
            printf(
                '<link rel="preload" as="style" href="%s">',
                esc_url($critical_css_url)
            );
        }
    }

    /**
     * Remove query strings from static resources
     */
    public function remove_query_strings($src) {
        if (strpos($src, '?ver=')) {
            $src = remove_query_arg('ver', $src);
        }
        return $src;
    }

    /**
     * Disable WordPress emojis
     */
    private function disable_emojis() {
        remove_action('wp_head', 'print_emoji_detection_script', 7);
        remove_action('admin_print_scripts', 'print_emoji_detection_script');
        remove_action('wp_print_styles', 'print_emoji_styles');
        remove_action('admin_print_styles', 'print_emoji_styles');
        remove_filter('the_content_feed', 'wp_staticize_emoji');
        remove_filter('comment_text_rss', 'wp_staticize_emoji');
        remove_filter('wp_mail', 'wp_staticize_emoji_for_email');

        add_filter('tiny_mce_plugins', function($plugins) {
            return is_array($plugins) ? array_diff($plugins, ['wpemoji']) : [];
        });

        add_filter('wp_resource_hints', function($urls, $relation_type) {
            if ($relation_type === 'dns-prefetch') {
                $urls = array_values(array_diff($urls, ['https://s.w.org']));
            }
            return $urls;
        }, 10, 2);
    }

    /**
     * Disable WordPress embeds
     */
    private function disable_embeds() {
        remove_action('rest_api_init', 'wp_oembed_register_route');
        remove_filter('oembed_dataparse', 'wp_filter_oembed_result', 10);
        remove_action('wp_head', 'wp_oembed_add_discovery_links');
        remove_action('wp_head', 'wp_oembed_add_host_js');
        remove_filter('pre_oembed_result', 'wp_filter_pre_oembed_result', 10);

        add_filter('embed_oembed_discover', '__return_false');
        add_filter('rewrite_rules_array', function($rules) {
            foreach ($rules as $rule => $rewrite) {
                if (strpos($rewrite, 'embed=true') !== false) {
                    unset($rules[$rule]);
                }
            }
            return $rules;
        });
    }

    /**
     * Minify CSS
     */
    private function minify_css($css) {
        // Remove comments
        $css = preg_replace('!/\*[^*]*\*+([^/][^*]*\*+)*/!', '', $css);

        // Remove whitespace
        $css = str_replace(["\r\n", "\r", "\n", "\t"], '', $css);
        $css = preg_replace('/\s+/', ' ', $css);

        // Remove unnecessary spaces
        $css = str_replace([' {', '{ ', ' }', '} ', ': ', ' ;'], ['{', '{', '}', '}', ':', ';'], $css);

        return trim($css);
    }

    /**
     * Minify JavaScript
     */
    private function minify_js($js) {
        // Basic minification (use with caution)
        // For production, use a proper JS minifier

        // Remove single-line comments
        $js = preg_replace('/\/\/[^\n]*/', '', $js);

        // Remove multi-line comments
        $js = preg_replace('/\/\*[\s\S]*?\*\//', '', $js);

        // Remove excessive whitespace
        $js = preg_replace('/\s+/', ' ', $js);

        // Remove whitespace around operators
        $js = preg_replace('/\s*([=+\-*\/,;{}()])\s*/', '$1', $js);

        return trim($js);
    }

    /**
     * Convert URL to file path
     */
    private function url_to_path($url) {
        // Handle protocol-relative URLs
        if (strpos($url, '//') === 0) {
            $url = 'https:' . $url;
        }

        // Handle relative URLs
        if (strpos($url, '/') === 0) {
            $url = home_url($url);
        }

        // Check if it's a local URL
        if (strpos($url, home_url()) !== 0) {
            return false;
        }

        // Convert to path
        $path = str_replace(home_url(), ABSPATH, $url);
        $path = str_replace('/', DIRECTORY_SEPARATOR, $path);

        return $path;
    }

    /**
     * Get image dimensions
     */
    private function get_image_dimensions($url) {
        // Cache dimensions
        $cache_key = 'vssv_img_dim_' . md5($url);
        $cached = get_transient($cache_key);

        if ($cached !== false) {
            return $cached;
        }

        $path = $this->url_to_path($url);
        if (!$path || !file_exists($path)) {
            return false;
        }

        $size = @getimagesize($path);
        if (!$size) {
            return false;
        }

        $dimensions = [
            'width' => $size[0],
            'height' => $size[1]
        ];

        // Cache for 30 days
        set_transient($cache_key, $dimensions, 30 * DAY_IN_SECONDS);

        return $dimensions;
    }

    /**
     * Run optimization
     */
    public function optimize() {
        $results = [
            'html_minified' => $this->settings['minify_html'],
            'css_optimized' => $this->settings['critical_css'],
            'js_deferred' => $this->settings['defer_js'],
            'lazy_loading' => $this->settings['lazy_load'],
            'resource_hints' => $this->settings['resource_hints']
        ];

        // Clear optimization cache
        global $wpdb;
        $wpdb->query(
            $wpdb->prepare(
                "DELETE FROM {$wpdb->options} WHERE option_name LIKE %s",
                $wpdb->esc_like('_transient_vssv_critical_') . '%'
            )
        );

        return $results;
    }

    /**
     * Get optimization recommendations
     */
    public function get_recommendations() {
        $recommendations = [];

        if (!$this->settings['minify_html']) {
            $recommendations[] = [
                'type' => 'warning',
                'message' => 'HTML minification is disabled',
                'action' => 'Enable HTML minification to reduce page size'
            ];
        }

        if (!$this->settings['critical_css']) {
            $recommendations[] = [
                'type' => 'warning',
                'message' => 'Critical CSS is not enabled',
                'action' => 'Enable critical CSS for faster initial rendering'
            ];
        }

        if (!$this->settings['lazy_load']) {
            $recommendations[] = [
                'type' => 'info',
                'message' => 'Lazy loading is disabled',
                'action' => 'Enable lazy loading for images and iframes'
            ];
        }

        return $recommendations;
    }
}