<?php
/**
 * Minified Asset Helper
 * Automatically uses .min.js/.min.css versions when available in production
 * Falls back to original files in development or if minified versions don't exist
 *
 * @package VS_Canvas
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Get the appropriate asset URL (minified if available in production)
 *
 * @param string $base_url   Plugin URL base (e.g., VSC_PLUGIN_URL)
 * @param string $base_dir   Plugin directory path (e.g., VSC_PLUGIN_DIR)
 * @param string $asset_path Relative path to asset (e.g., 'assets/js/scripts.js')
 * @return string Full URL to the asset (minified or original)
 */
if (!function_exists('vsc_get_asset_url')) {
function vsc_get_asset_url($base_url, $base_dir, $asset_path) {
    // Check if we should use minified versions
    // Use minified in production (when SCRIPT_DEBUG is false)
    $use_minified = !defined('SCRIPT_DEBUG') || !SCRIPT_DEBUG;

    if ($use_minified) {
        // Generate minified path
        $ext = pathinfo($asset_path, PATHINFO_EXTENSION);
        $min_path = preg_replace('/\.' . $ext . '$/', '.min.' . $ext, $asset_path);
        $min_file = $base_dir . $min_path;

        // Use minified if it exists
        if (file_exists($min_file)) {
            return $base_url . $min_path;
        }
    }

    // Fall back to original
    return $base_url . $asset_path;
}
}

/**
 * Enqueue a style with automatic minification detection
 *
 * @param string $handle     Style handle
 * @param string $base_url   Plugin URL base
 * @param string $base_dir   Plugin directory path
 * @param string $asset_path Relative path to CSS file
 * @param array  $deps       Dependencies
 * @param string $version    Version string
 * @param string $media      Media type
 */
if (!function_exists('vsc_enqueue_style')) {
function vsc_enqueue_style($handle, $base_url, $base_dir, $asset_path, $deps = array(), $version = '', $media = 'all') {
    $url = vsc_get_asset_url($base_url, $base_dir, $asset_path);
    wp_enqueue_style($handle, $url, $deps, $version, $media);
}
}

/**
 * Enqueue a script with automatic minification detection
 *
 * @param string $handle     Script handle
 * @param string $base_url   Plugin URL base
 * @param string $base_dir   Plugin directory path
 * @param string $asset_path Relative path to JS file
 * @param array  $deps       Dependencies
 * @param string $version    Version string
 * @param bool   $in_footer  Load in footer
 */
if (!function_exists('vsc_enqueue_script')) {
function vsc_enqueue_script($handle, $base_url, $base_dir, $asset_path, $deps = array(), $version = '', $in_footer = true) {
    $url = vsc_get_asset_url($base_url, $base_dir, $asset_path);
    wp_enqueue_script($handle, $url, $deps, $version, $in_footer);
}
}
