/**
 * VS Site Vector - Admin UI Scripts
 * Main JavaScript for the HTMX admin interface
 *
 * @package VS_Site_Vector
 * @version 2.1.0
 */
(function() {
    'use strict';

    // ============================================
    // VSSV Namespace - All public functions
    // ============================================
    var VSSV = window.VSSV = {};

    /**
     * Escape HTML for safe display
     */
    function escapeHtml(text) {
        if (!text) return '';
        var el = document.createElement('div');
        el.textContent = text;
        return el.innerHTML;
    }

    function getNonce() {
        return (typeof vssvData !== 'undefined') ? vssvData.ajaxNonce : '';
    }

    /**
     * Show toast notification
     */
    VSSV.toast = function(title, message, type) {
        type = type || 'info';
        var container = document.getElementById('vs-toast-container');
        if (!container) return;

        var toast = document.createElement('div');
        toast.className = 'vs-toast ' + type;

        toast.innerHTML = '<div class="vs-toast-content">' +
            (title ? '<div class="vs-toast-title">' + escapeHtml(title) + '</div>' : '') +
            (message ? '<div class="vs-toast-message">' + escapeHtml(message) + '</div>' : '') +
            '</div><button class="vs-toast-close">&times;</button>';

        toast.querySelector('.vs-toast-close').onclick = function() {
            removeToast(toast);
        };

        container.appendChild(toast);
        setTimeout(function() { removeToast(toast); }, 5000);

        function removeToast(el) {
            el.classList.add('removing');
            setTimeout(function() {
                if (el.parentNode) el.remove();
            }, 300);
        }
    };

    // Legacy alias
    window.vssvShowToast = VSSV.toast;

    /**
     * Quick action handlers
     */
    VSSV.quickAction = function(action) {
        if (!confirm('Run ' + action.replace(/-/g, ' ') + '?')) return;

        var xhr = new XMLHttpRequest();
        xhr.open('POST', ajaxurl, true);
        xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');

        xhr.onload = function() {
            if (xhr.status === 200) {
                try {
                    var response = JSON.parse(xhr.responseText);
                    if (response.success) {
                        VSSV.toast('Success', action + ' completed', 'success');
                    } else {
                        VSSV.toast('Error', response.data || 'Action failed', 'error');
                    }
                } catch (e) {
                    VSSV.toast('Error', 'Invalid response', 'error');
                }
            }
        };

        xhr.send('action=vssv_quick_action&quick_action=' + action + '&nonce=' + getNonce());
    };

    window.vssv_quickAction = VSSV.quickAction;

    /**
     * Handle optimization button
     */
    VSSV.handleOptimization = function(btn) {
        btn.disabled = true;
        var originalText = btn.innerHTML;
        btn.innerHTML = '<i data-lucide="loader" style="width:16px;height:16px;margin-right:6px;animation:spin 1s linear infinite;"></i>Optimizing...';

        var xhr = new XMLHttpRequest();
        xhr.open('POST', ajaxurl, true);
        xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');

        xhr.onload = function() {
            btn.disabled = false;
            btn.innerHTML = originalText;

            if (xhr.status === 200) {
                try {
                    var response = JSON.parse(xhr.responseText);
                    if (response.success) {
                        VSSV.toast('Success', 'Optimization complete! Score: ' + response.data.score, 'success');
                    } else {
                        VSSV.toast('Error', 'Optimization failed', 'error');
                    }
                } catch (e) {
                    VSSV.toast('Error', 'Invalid response', 'error');
                }
            }

            // Re-create icons after restoring button HTML
            if (typeof lucide !== 'undefined') {
                lucide.createIcons();
            }
        };

        xhr.send('action=vssv_run_optimization&nonce=' + getNonce());
    };

    window.vssv_handleOptimization = VSSV.handleOptimization;

    /**
     * Clear cache
     */
    VSSV.clearCache = function() {
        if (!confirm('Clear all caches?')) return;

        var xhr = new XMLHttpRequest();
        xhr.open('POST', ajaxurl, true);
        xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');

        xhr.onload = function() {
            if (xhr.status === 200) {
                try {
                    var response = JSON.parse(xhr.responseText);
                    if (response.success) {
                        VSSV.toast('Success', 'All caches cleared', 'success');
                    }
                } catch (e) {
                    VSSV.toast('Error', 'Invalid response', 'error');
                }
            }
        };

        xhr.send('action=vssv_clear_cache&nonce=' + getNonce());
    };

    window.vssv_clearCache = VSSV.clearCache;

    /**
     * Generate static site
     */
    VSSV.generateStatic = function() {
        if (!confirm('Generate static site? This may take several minutes.')) return;
        VSSV.toast('Info', 'Starting static generation...', 'info');
    };

    window.vssv_generateStatic = VSSV.generateStatic;

    /**
     * Reset settings
     */
    VSSV.resetSettings = function() {
        if (!confirm('Reset all settings to defaults?')) return;
        VSSV.toast('Info', 'Settings reset', 'info');
    };

    window.vssv_resetSettings = VSSV.resetSettings;

    /**
     * Save all settings (used by settings.php and optimization.php)
     */
    VSSV.saveAllSettings = function() {
        var settings = {};
        document.querySelectorAll('[data-option]').forEach(function(el) {
            var option = el.dataset.option;
            var value = el.type === 'checkbox' ? el.checked : el.value;
            settings[option] = value;
        });

        var xhr = new XMLHttpRequest();
        xhr.open('POST', ajaxurl, true);
        xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');
        xhr.onload = function() {
            if (xhr.status === 200) {
                try {
                    var response = JSON.parse(xhr.responseText);
                    if (response.success) {
                        VSSV.toast('Success', 'Settings saved successfully', 'success');
                    } else {
                        VSSV.toast('Error', response.data || 'Failed to save settings', 'error');
                    }
                } catch (e) {
                    VSSV.toast('Error', 'Invalid response', 'error');
                }
            }
        };
        xhr.send('action=vssv_save_settings&settings=' + encodeURIComponent(JSON.stringify(settings)) + '&nonce=' + getNonce());
    };

    window.vssv_saveAllSettings = VSSV.saveAllSettings;
    window.vssv_saveOptimizationSettings = VSSV.saveAllSettings;

    /**
     * Clear Web Vitals data
     */
    VSSV.clearVitalsData = function() {
        if (!confirm('Are you sure you want to delete ALL Web Vitals data? This cannot be undone.')) return;

        var xhr = new XMLHttpRequest();
        xhr.open('POST', ajaxurl, true);
        xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');
        xhr.onload = function() {
            if (xhr.status === 200) {
                try {
                    var response = JSON.parse(xhr.responseText);
                    if (response.success) {
                        VSSV.toast('Success', 'All Web Vitals data cleared', 'success');
                    } else {
                        VSSV.toast('Error', response.data || 'Failed to clear data', 'error');
                    }
                } catch (e) {
                    VSSV.toast('Error', 'Invalid response', 'error');
                }
            }
        };
        xhr.send('action=vssv_clear_vitals_data&nonce=' + getNonce());
    };

    window.vssv_clearVitalsData = VSSV.clearVitalsData;

    // ============================================
    // Core Initialization
    // ============================================

    var initialized = false;

    function initPage() {
        if (initialized) return;
        initialized = true;
        // Initialize Lucide icons
        if (typeof lucide !== 'undefined') {
            lucide.createIcons();
        }

        // Handle active nav state
        var navTabs = document.querySelectorAll('.vs-nav-tab');
        navTabs.forEach(function(tab) {
            tab.addEventListener('click', function() {
                navTabs.forEach(function(t) {
                    t.classList.remove('active');
                });
                this.classList.add('active');
            });
        });
    }

    // Single initialization point - DOMContentLoaded only
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', initPage);
    } else {
        initPage();
    }

    // Re-initialize after HTMX swap
    document.addEventListener('htmx:afterSwap', function() {
        initialized = false;
        requestAnimationFrame(initPage);
    });
})();
