<?php
/**
 * Plugin Name: VS Site Vector
 * Plugin URI: https://veroscale.au/plugins/vs-site-vector
 * Description: The complete WordPress performance suite. Live optimization, CDN deployment, and comprehensive caching - everything you need for maximum speed in one powerful plugin.
 * Version: 1.3.0
 * Author: Sabastian Nguyen - Vero Scale
 * Author URI: https://veroscale.au
 * License: GPL v2 or later
 * Text Domain: vs-site-vector
 * Domain Path: /languages
 * Requires at least: 5.8
 * Requires PHP: 7.4
 * Network: true
 *
 * Features:
 * - Advanced multi-layer caching
 * - Multi-platform CDN deployment
 * - Real-time performance monitoring
 * - Complete asset optimization
 * - Database optimization
 * - And much more!
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Plugin constants
define('VSSV_VERSION', '1.3.0');
define('VSSV_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('VSSV_PLUGIN_URL', plugin_dir_url(__FILE__));
define('VSSV_PLUGIN_BASENAME', plugin_basename(__FILE__));
define('VSSV_CACHE_DIR', WP_CONTENT_DIR . '/cache/vs-site-vector/');
define('VSSV_CACHE_URL', WP_CONTENT_URL . '/cache/vs-site-vector/');

// Performance benchmarking
if (!defined('VSSV_START_TIME')) {
    define('VSSV_START_TIME', microtime(true));
}

/**
 * Main VS Site Vector Class
 */
class VS_Site_Vector {

    /**
     * Instance of this class
     */
    private static $instance = null;

    /**
     * Performance modules
     */
    public $modules = [];

    /**
     * Current performance score
     */
    private $performance_score = 0;

    /**
     * Get singleton instance
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        // Load required files
        $this->load_dependencies();

        // Initialize modules
        $this->init_modules();

        // Setup hooks
        $this->setup_hooks();

        // Check environment
        $this->check_environment();
    }

    /**
     * Load plugin dependencies
     */
    private function load_dependencies() {
        // Configuration class (load first)
        require_once VSSV_PLUGIN_DIR . 'includes/class-vssv-config.php';

        // Core classes
        require_once VSSV_PLUGIN_DIR . 'includes/class-vssv-optimizer.php';
        require_once VSSV_PLUGIN_DIR . 'includes/class-vssv-cache.php';
        require_once VSSV_PLUGIN_DIR . 'includes/class-vssv-deploy.php';
        require_once VSSV_PLUGIN_DIR . 'includes/class-vssv-api.php';

        // Diagnostics class (always load - handles its own conditional execution)
        require_once VSSV_PLUGIN_DIR . 'includes/class-vssv-diagnostics.php';

        // Admin classes
        if (is_admin()) {
            require_once VSSV_PLUGIN_DIR . 'admin/class-vssv-admin.php';
            require_once VSSV_PLUGIN_DIR . 'admin/class-vssv-dashboard.php';

            // HTMX Modern UI (optional modern view)
            require_once VSSV_PLUGIN_DIR . 'admin/htmx/class-vssv-toggle-handler.php';
            require_once VSSV_PLUGIN_DIR . 'admin/htmx/class-vssv-page-registry.php';
            require_once VSSV_PLUGIN_DIR . 'admin/htmx/class-vssv-htmx-loader.php';
        }
    }

    /**
     * Initialize optimization modules
     */
    private function init_modules() {
        // Check for emergency disable or safe mode
        if (VSSV_Config::is_emergency_disabled()) {
            add_action('admin_notices', function() {
                echo '<div class="notice notice-warning"><p><strong>VS Site Vector:</strong> Emergency disable mode is active. All optimizations are bypassed.</p></div>';
            });
            return;
        }

        if (VSSV_Config::is_safe_mode()) {
            add_action('admin_notices', function() {
                echo '<div class="notice notice-info"><p><strong>VS Site Vector:</strong> Safe mode is active. Only essential features are running.</p></div>';
            });
            // Only load essential modules in safe mode
            $this->modules['cache'] = new VSSV_Cache();
            return;
        }

        // Get active modules from settings
        $active_modules = get_option('vssv_active_modules', $this->get_default_modules());

        // DEBUG: Check if test mode is enabled via constant
        $test_mode = defined('VSSV_TEST_MODE') && VSSV_TEST_MODE;
        $disabled_modules = defined('VSSV_DISABLED_MODULES') ? VSSV_DISABLED_MODULES : [];

        if ($test_mode) {
            error_log('VSSV: TEST MODE ENABLED - Active modules: ' . json_encode($active_modules));
            error_log('VSSV: Disabled modules: ' . json_encode($disabled_modules));
        }

        // Core optimizer (always active unless disabled)
        if (!in_array('optimizer', $disabled_modules)) {
            try {
                $this->modules['optimizer'] = new VSSV_Optimizer();
                if ($test_mode) error_log('VSSV: Optimizer loaded successfully');
            } catch (Exception $e) {
                error_log('VSSV ERROR: Optimizer failed - ' . $e->getMessage());
            }
        } else if ($test_mode) {
            error_log('VSSV: Optimizer DISABLED (test mode)');
        }

        // Cache module
        if (in_array('cache', $active_modules) && !in_array('cache', $disabled_modules)) {
            try {
                $this->modules['cache'] = new VSSV_Cache();
                if ($test_mode) error_log('VSSV: Cache module loaded successfully');
            } catch (Exception $e) {
                error_log('VSSV ERROR: Cache module failed - ' . $e->getMessage());
            }
        } else if ($test_mode && in_array('cache', $disabled_modules)) {
            error_log('VSSV: Cache module DISABLED (test mode)');
        }

        // Deployment management
        if (in_array('deploy', $active_modules) && !in_array('deploy', $disabled_modules)) {
            try {
                $this->modules['deploy'] = new VSSV_Deploy();
                if ($test_mode) error_log('VSSV: Deploy module loaded successfully');
            } catch (Exception $e) {
                error_log('VSSV ERROR: Deploy module failed - ' . $e->getMessage());
            }
        } else if ($test_mode && in_array('deploy', $disabled_modules)) {
            error_log('VSSV: Deploy module DISABLED (test mode)');
        }

        // REST API optimizations
        if (in_array('api', $active_modules) && !in_array('api', $disabled_modules)) {
            try {
                $this->modules['api'] = new VSSV_API();
                if ($test_mode) error_log('VSSV: API module loaded successfully');
            } catch (Exception $e) {
                error_log('VSSV ERROR: API module failed - ' . $e->getMessage());
            }
        } else if ($test_mode && in_array('api', $disabled_modules)) {
            error_log('VSSV: API module DISABLED (test mode)');
        }

        // Diagnostics (always load but can be disabled)
        if (!in_array('diagnostics', $disabled_modules)) {
            try {
                if ($test_mode) error_log('VSSV: Loading Diagnostics module');
                VSSV_Diagnostics::get_instance();
                if ($test_mode) error_log('VSSV: Diagnostics module loaded successfully');
            } catch (Exception $e) {
                error_log('VSSV ERROR: Diagnostics module failed - ' . $e->getMessage());
            }
        } else if ($test_mode) {
            error_log('VSSV: Diagnostics module DISABLED (test mode)');
        }

        // Admin interface
        if (is_admin()) {
            // Load the dashboard with inline HTML/CSS/JS
            if (!in_array('admin', $disabled_modules)) {
                try {
                    $this->modules['admin'] = new VSSV_Admin();
                    if ($test_mode) error_log('VSSV: Admin module loaded successfully');
                } catch (Exception $e) {
                    error_log('VSSV ERROR: Admin module failed - ' . $e->getMessage());
                }
            } else if ($test_mode) {
                error_log('VSSV: Admin module DISABLED (test mode)');
            }

            if (!in_array('dashboard', $disabled_modules)) {
                try {
                    $this->modules['dashboard'] = new VSSV_Dashboard();
                    if ($test_mode) error_log('VSSV: Dashboard module loaded successfully');
                } catch (Exception $e) {
                    error_log('VSSV ERROR: Dashboard module failed - ' . $e->getMessage());
                }
            } else if ($test_mode) {
                error_log('VSSV: Dashboard module DISABLED (test mode)');
            }

            // Initialize HTMX modern UI loader
            if (!in_array('htmx_loader', $disabled_modules)) {
                try {
                    $this->modules['htmx_loader'] = new VSSV_HTMX_Loader();

                    // Wire HTMX loader to dashboard
                    if (isset($this->modules['dashboard'])) {
                        $this->modules['dashboard']->set_htmx_loader($this->modules['htmx_loader']);
                    }

                    if ($test_mode) error_log('VSSV: HTMX Loader module loaded successfully');
                } catch (Exception $e) {
                    error_log('VSSV ERROR: HTMX Loader module failed - ' . $e->getMessage());
                }
            } else if ($test_mode) {
                error_log('VSSV: HTMX Loader module DISABLED (test mode)');
            }
        }

        if ($test_mode) {
            error_log('VSSV: Module initialization complete');
        }
    }

    /**
     * Setup WordPress hooks
     */
    private function setup_hooks() {
        // Activation/Deactivation
        register_activation_hook(VSSV_PLUGIN_BASENAME, [$this, 'activate']);
        register_deactivation_hook(VSSV_PLUGIN_BASENAME, [$this, 'deactivate']);

        // Core actions
        add_action('init', [$this, 'init'], 0);
        add_action('plugins_loaded', [$this, 'load_textdomain']);
        add_action('wp_loaded', [$this, 'wp_loaded']);

        // Performance optimizations
        add_action('send_headers', [$this, 'set_browser_cache_headers'], 0);
        add_action('template_redirect', [$this, 'start_optimization'], -999);
        add_action('shutdown', [$this, 'finish_optimization'], 999);

        // Asset versioning for cache busting
        add_filter('style_loader_src', [$this, 'add_asset_version'], 10, 2);
        add_filter('script_loader_src', [$this, 'add_asset_version'], 10, 2);

        // AJAX handlers
        add_action('wp_ajax_vssv_run_optimization', [$this, 'ajax_run_optimization']);
        add_action('wp_ajax_vssv_clear_cache', [$this, 'ajax_clear_cache']);
        add_action('wp_ajax_vssv_get_performance_data', [$this, 'ajax_get_performance_data']);
        add_action('wp_ajax_vssv_get_dashboard_data', [$this, 'ajax_get_dashboard_data']);
        add_action('wp_ajax_vssv_quick_action', [$this, 'ajax_quick_action']);
        add_action('wp_ajax_vssv_get_performance_history', [$this, 'ajax_get_performance_history']);
        add_action('wp_ajax_vssv_toggle_optimization', [$this, 'ajax_toggle_optimization']);

        // Public AJAX endpoint for Web Vitals collection (no auth required)
        add_action('wp_ajax_vssv_collect_vitals', [$this, 'ajax_collect_vitals']);
        add_action('wp_ajax_nopriv_vssv_collect_vitals', [$this, 'ajax_collect_vitals']);

        // Cron jobs
        add_action('vssv_weekly_cleanup', [$this, 'run_weekly_cleanup']);

        // Ensure database tables exist (in case plugin was updated without reactivation)
        add_action('admin_init', [$this, 'maybe_create_tables']);

        // REST API endpoints
        add_action('rest_api_init', [$this, 'register_rest_routes']);

        // Frontend performance tracker (no cookies - uses localStorage)
        add_action('wp_enqueue_scripts', [$this, 'enqueue_performance_tracker']);
    }

    /**
     * Enqueue performance tracker script on frontend
     * Collects real Web Vitals metrics and sends to server
     * Note: wp_enqueue_scripts only fires on frontend, so no is_admin() check needed
     */
    public function enqueue_performance_tracker() {
        // Use minified version in production
        $script_suffix = defined('SCRIPT_DEBUG') && SCRIPT_DEBUG ? '' : '.min';
        wp_enqueue_script(
            'vssv-performance-tracker',
            VSSV_PLUGIN_URL . 'assets/js/vssv-performance-tracker' . $script_suffix . '.js',
            [],
            VSSV_VERSION,
            true // Load in footer for better performance
        );

        // Pass configuration to the script
        wp_add_inline_script(
            'vssv-performance-tracker',
            'window.vssvTrackerConfig = ' . json_encode([
                'ajaxUrl' => admin_url('admin-ajax.php'),
                'token' => wp_create_nonce('vssv_vitals_' . date('Y-m-d')),
                'debug' => defined('WP_DEBUG') && WP_DEBUG,
            ]) . ';',
            'before'
        );
    }

    /**
     * Add plugin version to asset URLs for cache busting
     *
     * @param string $src Asset source URL
     * @param string $handle Asset handle
     * @return string Modified source URL with version parameter
     */
    public function add_asset_version($src, $handle) {
        // Only add version to plugin assets or theme assets
        if (strpos($src, VSSV_PLUGIN_URL) !== false || strpos($src, get_stylesheet_directory_uri()) !== false) {
            // Remove existing version parameter if present
            $src = remove_query_arg('ver', $src);

            // Build version string with multiple factors for cache busting:
            // 1. Plugin version
            // 2. File modification time
            // 3. Cache clear timestamp (when cache is cleared, this changes)
            $cache_cleared_at = get_option('vssv_cache_cleared_at', 0);
            $version = VSSV_VERSION . '.' . ($cache_cleared_at ? $cache_cleared_at : time());

            // Add cache buster version
            $src = add_query_arg('ver', $version, $src);
        }

        return $src;
    }

    /**
     * Check server environment
     */
    private function check_environment() {
        $requirements = [];

        // PHP version
        if (version_compare(PHP_VERSION, '7.4', '<')) {
            $requirements[] = sprintf('PHP 7.4 or higher (current: %s)', PHP_VERSION);
        }

        // Required PHP extensions
        $required_extensions = ['mbstring', 'dom', 'xml', 'json'];
        foreach ($required_extensions as $ext) {
            if (!extension_loaded($ext)) {
                $requirements[] = sprintf('PHP %s extension', $ext);
            }
        }

        // Recommended PHP extensions
        $recommended_extensions = ['imagick', 'redis', 'apcu', 'opcache'];
        foreach ($recommended_extensions as $ext) {
            if (!extension_loaded($ext)) {
                update_option('vssv_recommended_' . $ext, false);
            } else {
                update_option('vssv_recommended_' . $ext, true);
            }
        }

        // Cache directory writable
        if (!is_writable(WP_CONTENT_DIR)) {
            $requirements[] = 'Write permissions for wp-content directory';
        }

        // Store requirements
        if (!empty($requirements)) {
            update_option('vssv_requirements_missing', $requirements);
            if (is_admin()) {
                add_action('admin_notices', [$this, 'show_requirements_notice']);
            }
        } else {
            delete_option('vssv_requirements_missing');
        }
    }

    /**
     * Plugin initialization
     */
    public function init() {
        // Create cache directories
        $this->create_cache_directories();

        // Schedule cron events
        $this->schedule_events();

        // Initialize optimization rules
        $this->init_optimization_rules();
    }


    /**
     * Create cache directories
     */
    private function create_cache_directories() {
        $directories = [
            VSSV_CACHE_DIR,
            VSSV_CACHE_DIR . 'html/',
            VSSV_CACHE_DIR . 'css/',
            VSSV_CACHE_DIR . 'js/',
            VSSV_CACHE_DIR . 'images/',
            VSSV_CACHE_DIR . 'critical/',
            VSSV_CACHE_DIR . 'fragments/'
        ];

        foreach ($directories as $dir) {
            if (!file_exists($dir)) {
                wp_mkdir_p($dir);
            }

            // Add/update .htaccess for security and caching
            $htaccess = $dir . '.htaccess';
            $htaccess_content = "# VS Site Vector Cache Directory\n";
            $htaccess_content .= "Options -Indexes\n";
            $htaccess_content .= "Deny from all\n\n";

            // Allow direct access to cached files (they should be served publicly)
            if ($dir === VSSV_CACHE_DIR . 'html/' || $dir === VSSV_CACHE_DIR . 'fragments/') {
                $htaccess_content .= "# Allow serving cached HTML\n";
                $htaccess_content .= "<FilesMatch \"\.html?$\">\n";
                $htaccess_content .= "  Allow from all\n";
                $htaccess_content .= "  <IfModule mod_headers.c>\n";
                $htaccess_content .= "    Header set Cache-Control \"public, max-age=3600\"\n";
                $htaccess_content .= "  </IfModule>\n";
                $htaccess_content .= "</FilesMatch>\n";
            }

            file_put_contents($htaccess, $htaccess_content);
        }
    }

    /**
     * Set browser cache headers for frontend responses
     */
    public function set_browser_cache_headers() {
        // Only on frontend
        if (is_admin() || wp_doing_ajax() || wp_doing_cron()) {
            return;
        }

        // Check if headers already sent
        if (headers_sent()) {
            return;
        }

        // Set default cache headers (1 hour for pages, longer for assets)
        if (is_singular() || is_home() || is_archive()) {
            // Pages/posts: 1 hour in browser, 24 hours in CDN
            header('Cache-Control: public, max-age=3600');
            header('CDN-Cache-Control: max-age=86400');
        } else {
            // Other pages: 30 minutes
            header('Cache-Control: public, max-age=1800');
        }

        // Add Last-Modified header
        header('Last-Modified: ' . gmdate('D, d M Y H:i:s', time()) . ' GMT');
    }

    /**
     * Start optimization process
     */
    public function start_optimization() {
        if (is_admin() || wp_doing_ajax() || wp_doing_cron()) {
            return;
        }

        // Start output buffering for HTML optimization
        ob_start([$this, 'optimize_output']);
    }

    /**
     * Optimize output HTML
     */
    public function optimize_output($html) {
        if (empty($html)) {
            return $html;
        }

        // Apply optimizations
        if (isset($this->modules['optimizer'])) {
            $html = $this->modules['optimizer']->optimize_html($html);
        }

        return $html;
    }

    /**
     * Finish optimization and cleanup
     */
    public function finish_optimization() {
        // Cleanup transients
        $this->cleanup_expired_transients();
    }

    /**
     * Calculate current performance score
     */
    public function calculate_performance_score() {
        $score = 100;
        $factors = [];

        // Page load time (0-40 points)
        $load_time = get_transient('vssv_avg_load_time') ?: 3.0;
        if ($load_time < 1.0) {
            $factors['load_time'] = 40;
        } elseif ($load_time < 2.0) {
            $factors['load_time'] = 35;
        } elseif ($load_time < 3.0) {
            $factors['load_time'] = 25;
        } else {
            $factors['load_time'] = max(0, 40 - ($load_time * 5));
        }

        // Cache hit rate (0-20 points)
        $cache_hits = get_transient('vssv_cache_hit_rate') ?: 0;
        $factors['cache'] = min(20, $cache_hits * 0.2);

        // Database queries (0-15 points)
        $db_queries = get_transient('vssv_avg_queries') ?: 50;
        if ($db_queries < 20) {
            $factors['database'] = 15;
        } elseif ($db_queries < 50) {
            $factors['database'] = 10;
        } else {
            $factors['database'] = max(0, 15 - ($db_queries / 10));
        }

        // Asset optimization (0-15 points)
        $assets_optimized = get_option('vssv_assets_optimized', false);
        $factors['assets'] = $assets_optimized ? 15 : 5;

        // Image optimization (0-10 points)
        $images_optimized = get_option('vssv_images_optimized', false);
        $factors['images'] = $images_optimized ? 10 : 3;

        // Calculate total
        $score = array_sum($factors);

        // Store for monitoring
        update_option('vssv_performance_score', $score);
        update_option('vssv_performance_factors', $factors);

        return round($score);
    }

    /**
     * Get default active modules
     */
    private function get_default_modules() {
        return [
            'cache',
            'deploy',
            'api'
        ];
    }

    /**
     * Plugin activation
     */
    public function activate() {
        // Create database tables
        $this->create_database_tables();

        // Set default options
        $this->set_default_options();

        // Schedule events
        $this->schedule_events();

        // Clear rewrite rules
        flush_rewrite_rules();

        // Log activation
        update_option('vssv_activated', time());
    }

    /**
     * Plugin deactivation
     */
    public function deactivate() {
        // Clear scheduled events
        wp_clear_scheduled_hook('vssv_weekly_cleanup');

        // Clear caches
        $this->clear_all_caches();

        // Log deactivation
        update_option('vssv_deactivated', time());
    }

    /**
     * Create database tables
     * Called on activation and also checked on admin_init
     */
    public function create_database_tables() {
        global $wpdb;

        $charset_collate = $wpdb->get_charset_collate();

        // Web Vitals metrics table (Real User Monitoring)
        $table_vitals = $wpdb->prefix . 'vssv_web_vitals';
        $sql_vitals = "CREATE TABLE IF NOT EXISTS $table_vitals (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            page_url varchar(255) NOT NULL,
            lcp float DEFAULT NULL,
            fid float DEFAULT NULL,
            cls float DEFAULT NULL,
            ttfb float DEFAULT NULL,
            fcp float DEFAULT NULL,
            dom_load float DEFAULT NULL,
            page_load float DEFAULT NULL,
            connection_type varchar(20) DEFAULT NULL,
            connection_speed float DEFAULT NULL,
            connection_rtt int DEFAULT NULL,
            device_type varchar(20) DEFAULT NULL,
            visitor_hash varchar(32) DEFAULT NULL,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY page_url (page_url(100)),
            KEY created_at (created_at),
            KEY visitor_hash (visitor_hash)
        ) $charset_collate;";

        // Legacy performance metrics table (keep for backwards compatibility)
        $table_metrics = $wpdb->prefix . 'vssv_metrics';
        $sql_metrics = "CREATE TABLE IF NOT EXISTS $table_metrics (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            metric_type varchar(50) NOT NULL,
            metric_value float NOT NULL,
            page_url varchar(255) DEFAULT NULL,
            user_agent varchar(255) DEFAULT NULL,
            timestamp datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY metric_type (metric_type),
            KEY timestamp (timestamp)
        ) $charset_collate;";

        // Optimization log table
        $table_log = $wpdb->prefix . 'vssv_optimization_log';
        $sql_log = "CREATE TABLE IF NOT EXISTS $table_log (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            optimization_type varchar(50) NOT NULL,
            before_value varchar(255) DEFAULT NULL,
            after_value varchar(255) DEFAULT NULL,
            improvement float DEFAULT NULL,
            timestamp datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY optimization_type (optimization_type),
            KEY timestamp (timestamp)
        ) $charset_collate;";

        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql_vitals);
        dbDelta($sql_metrics);
        dbDelta($sql_log);
    }

    /**
     * Check if tables exist and create/update them if needed
     * Called on admin_init to handle plugin updates without reactivation
     */
    public function maybe_create_tables() {
        global $wpdb;

        // Check version to see if we need to update tables
        $db_version = get_option('vssv_db_version', '0');
        $current_version = '2.1.0'; // Increment when schema changes

        if (version_compare($db_version, $current_version, '>=')) {
            return;
        }

        $table = $wpdb->prefix . 'vssv_web_vitals';
        $exists = $wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table));

        // Always run dbDelta to update schema (it's safe to run multiple times)
        $this->create_database_tables();

        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('VSSV: Updated database tables to version ' . $current_version);
        }

        update_option('vssv_db_version', $current_version);
    }

    /**
     * Set default options
     */
    private function set_default_options() {
        // Get safe defaults from config
        $config_defaults = VSSV_Config::get_defaults();

        // Add version and active modules
        $defaults = array_merge($config_defaults, [
            'vssv_version' => VSSV_VERSION,
            'vssv_active_modules' => $this->get_default_modules(),
        ]);

        // Apply environment-specific settings
        $environment = defined('WP_ENVIRONMENT_TYPE') ? WP_ENVIRONMENT_TYPE : 'production';
        $env_settings = VSSV_Config::get_environment_settings($environment);
        $defaults = array_merge($defaults, $env_settings);

        foreach ($defaults as $key => $value) {
            if (get_option($key) === false) {
                add_option($key, $value);
            }
        }

        // Apply compatibility settings
        $compat_settings = VSSV_Config::get_compatibility_settings();
        update_option('vssv_compatibility_settings', $compat_settings);
    }

    /**
     * Schedule cron events
     */
    private function schedule_events() {
        if (!wp_next_scheduled('vssv_weekly_cleanup')) {
            wp_schedule_event(time() + 86400, 'weekly', 'vssv_weekly_cleanup');
        }
    }

    /**
     * Clear all caches
     */
    public function clear_all_caches() {
        // Clear VS Site Vector caches
        if (isset($this->modules['cache'])) {
            $this->modules['cache']->clear_all();
        }

        // Clear WordPress object cache
        if (function_exists('wp_cache_flush')) {
            wp_cache_flush();
        }

        // Clear transients using WordPress function
        // This is safer than direct SQL queries
        delete_expired_transients(true); // Force delete all transients

        // If we need to use direct queries, use prepared statements
        global $wpdb;
        $wpdb->query(
            $wpdb->prepare(
                "DELETE FROM {$wpdb->options} WHERE option_name LIKE %s",
                $wpdb->esc_like('_transient_') . '%'
            )
        );
        $wpdb->query(
            $wpdb->prepare(
                "DELETE FROM {$wpdb->options} WHERE option_name LIKE %s",
                $wpdb->esc_like('_site_transient_') . '%'
            )
        );

        // Clear OPcache
        if (function_exists('opcache_reset')) {
            opcache_reset();
        }

        // Update cache clear timestamp (for browser cache invalidation)
        update_option('vssv_cache_cleared_at', time());

        // Trigger action for other plugins
        do_action('vssv_cache_cleared');

        return true;
    }

    /**
     * Load plugin text domain
     */
    public function load_textdomain() {
        load_plugin_textdomain(
            'vs-site-vector',
            false,
            dirname(VSSV_PLUGIN_BASENAME) . '/languages/'
        );
    }

    /**
     * WordPress loaded hook
     */
    public function wp_loaded() {
        // Late initialization for modules that need WordPress fully loaded
        do_action('vssv_loaded');
    }

    /**
     * Initialize optimization rules
     */
    private function init_optimization_rules() {
        // Load custom optimization rules
        $rules = get_option('vssv_optimization_rules', []);

        // Apply rules to modules
        foreach ($this->modules as $module) {
            if (method_exists($module, 'apply_rules')) {
                $module->apply_rules($rules);
            }
        }
    }

    /**
     * Run weekly cleanup tasks
     */
    public function run_weekly_cleanup() {
        // Clean old metrics
        global $wpdb;
        $wpdb->query(
            $wpdb->prepare(
                "DELETE FROM {$wpdb->prefix}vssv_metrics
                WHERE timestamp < %s",
                gmdate('Y-m-d H:i:s', strtotime('-30 days'))
            )
        );

        // Clean old logs
        $wpdb->query(
            $wpdb->prepare(
                "DELETE FROM {$wpdb->prefix}vssv_optimization_log
                WHERE timestamp < %s",
                gmdate('Y-m-d H:i:s', strtotime('-60 days'))
            )
        );

        // Rebuild cache
        if (isset($this->modules['cache'])) {
            $this->modules['cache']->rebuild();
        }
    }

    /**
     * AJAX handler for running optimization
     */
    public function ajax_run_optimization() {
        check_ajax_referer('vssv_ajax_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }

        $results = [];

        // Run each module's optimization
        foreach ($this->modules as $name => $module) {
            if (method_exists($module, 'optimize')) {
                $results[$name] = $module->optimize();
            }
        }

        // Clear caches
        $this->clear_all_caches();

        // Log optimization
        $this->log_optimization('manual', json_encode($results));

        wp_send_json_success([
            'message' => 'Optimization completed successfully',
            'results' => $results,
            'score' => $this->calculate_performance_score()
        ]);
    }

    /**
     * AJAX handler for clearing cache
     */
    public function ajax_clear_cache() {
        check_ajax_referer('vssv_ajax_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }

        $this->clear_all_caches();

        wp_send_json_success([
            'message' => 'All caches cleared successfully'
        ]);
    }

    /**
     * AJAX handler for toggling optimization features
     */
    public function ajax_toggle_optimization() {
        check_ajax_referer('vssv_ajax_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }

        $feature = isset($_POST['feature']) ? sanitize_text_field($_POST['feature']) : '';
        $enabled = isset($_POST['enabled']) ? ($_POST['enabled'] === 'true' || $_POST['enabled'] === '1') : false;

        if (empty($feature)) {
            wp_send_json_error(['message' => 'No feature specified']);
        }

        // Map feature keys to option names
        $option_map = [
            'cache'        => 'vssv_cache_enabled',
            'minify_html'  => 'vssv_minify_html',
            'minify_css'   => 'vssv_inline_small_css',
            'minify_js'    => 'vssv_inline_small_js',
            'lazy_load'    => 'vssv_lazy_load',
            'critical_css' => 'vssv_critical_css',
            'defer_js'     => 'vssv_defer_js',
            'browser_cache'=> 'vssv_browser_cache',
            'cdn'          => 'vssv_cdn_enabled',
            'gzip'         => 'vssv_gzip_enabled',
        ];

        if (!isset($option_map[$feature])) {
            wp_send_json_error(['message' => 'Invalid feature: ' . $feature]);
        }

        $option_name = $option_map[$feature];
        update_option($option_name, $enabled ? '1' : '0');

        // Clear caches after toggling
        $this->clear_all_caches();

        wp_send_json_success([
            'feature' => $feature,
            'enabled' => $enabled,
            'message' => $enabled ? 'Feature enabled' : 'Feature disabled'
        ]);
    }

    /**
     * AJAX handler for collecting Web Vitals from frontend
     * Public endpoint - no authentication required
     */
    public function ajax_collect_vitals() {
        global $wpdb;

        // Debug logging
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('VSSV: ajax_collect_vitals called');
            // Sanitize POST data before logging to prevent information disclosure
            $sanitized_post = array_map('sanitize_text_field', $_POST);
            error_log('VSSV: Sanitized POST data: ' . print_r($sanitized_post, true));
        }

        // Rate limiting: max 1 request per page load (use a simple token)
        $token = isset($_POST['token']) ? sanitize_text_field($_POST['token']) : '';
        if (empty($token)) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('VSSV: Missing token');
            }
            wp_send_json_error(['message' => 'Missing token']);
        }

        // Get and validate metrics
        $page_url = isset($_POST['url']) ? esc_url_raw($_POST['url']) : '';
        $lcp = isset($_POST['lcp']) ? floatval($_POST['lcp']) : null;
        $fid = isset($_POST['fid']) ? floatval($_POST['fid']) : null;
        $cls = isset($_POST['cls']) ? floatval($_POST['cls']) : null;
        $ttfb = isset($_POST['ttfb']) ? floatval($_POST['ttfb']) : null;
        $fcp = isset($_POST['fcp']) ? floatval($_POST['fcp']) : null;
        $dom_load = isset($_POST['dom_load']) ? floatval($_POST['dom_load']) : null;
        $page_load = isset($_POST['page_load']) ? floatval($_POST['page_load']) : null;
        $connection_type = isset($_POST['connection_type']) ? sanitize_text_field($_POST['connection_type']) : null;
        $connection_speed = isset($_POST['connection_speed']) ? floatval($_POST['connection_speed']) : null;
        $connection_rtt = isset($_POST['connection_rtt']) ? intval($_POST['connection_rtt']) : null;
        $device_type = isset($_POST['device']) ? sanitize_text_field($_POST['device']) : null;

        // Validate - must have at least URL and one metric
        if (empty($page_url)) {
            wp_send_json_error(['message' => 'Missing URL']);
        }

        if ($lcp === null && $fcp === null && $ttfb === null) {
            wp_send_json_error(['message' => 'No metrics provided']);
        }

        // Sanity checks on values (reject obviously invalid data)
        if ($lcp !== null && ($lcp < 0 || $lcp > 60000)) $lcp = null; // Max 60 seconds
        if ($fid !== null && ($fid < 0 || $fid > 10000)) $fid = null; // Max 10 seconds
        if ($cls !== null && ($cls < 0 || $cls > 10)) $cls = null; // Max CLS of 10
        if ($ttfb !== null && ($ttfb < 0 || $ttfb > 30000)) $ttfb = null; // Max 30 seconds
        if ($fcp !== null && ($fcp < 0 || $fcp > 60000)) $fcp = null;
        if ($dom_load !== null && ($dom_load < 0 || $dom_load > 120000)) $dom_load = null;
        if ($page_load !== null && ($page_load < 0 || $page_load > 120000)) $page_load = null;

        // Create anonymous visitor hash (for deduplication, not tracking)
        // Based on IP + User Agent + Date (changes daily)
        $visitor_hash = md5(
            $_SERVER['REMOTE_ADDR'] .
            ($_SERVER['HTTP_USER_AGENT'] ?? '') .
            date('Y-m-d')
        );

        // Check for duplicate (same visitor, same page, within last 5 minutes)
        $table = $wpdb->prefix . 'vssv_web_vitals';
        $recent = $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM $table
             WHERE visitor_hash = %s
             AND page_url = %s
             AND created_at > DATE_SUB(NOW(), INTERVAL 5 MINUTE)
             LIMIT 1",
            $visitor_hash,
            $page_url
        ));

        if ($recent) {
            // Update existing record instead of creating duplicate
            $wpdb->update(
                $table,
                [
                    'lcp' => $lcp,
                    'fid' => $fid,
                    'cls' => $cls,
                    'ttfb' => $ttfb,
                    'fcp' => $fcp,
                    'dom_load' => $dom_load,
                    'page_load' => $page_load,
                ],
                ['id' => $recent],
                ['%f', '%f', '%f', '%f', '%f', '%f', '%f'],
                ['%d']
            );
            wp_send_json_success(['message' => 'Updated', 'id' => $recent]);
        }

        // Build data and format arrays dynamically to handle NULL values
        $data = ['page_url' => $page_url];
        $format = ['%s'];

        if ($lcp !== null) { $data['lcp'] = $lcp; $format[] = '%f'; }
        if ($fid !== null) { $data['fid'] = $fid; $format[] = '%f'; }
        if ($cls !== null) { $data['cls'] = $cls; $format[] = '%f'; }
        if ($ttfb !== null) { $data['ttfb'] = $ttfb; $format[] = '%f'; }
        if ($fcp !== null) { $data['fcp'] = $fcp; $format[] = '%f'; }
        if ($dom_load !== null) { $data['dom_load'] = $dom_load; $format[] = '%f'; }
        if ($page_load !== null) { $data['page_load'] = $page_load; $format[] = '%f'; }
        if ($connection_type !== null) { $data['connection_type'] = $connection_type; $format[] = '%s'; }
        if ($connection_speed !== null) { $data['connection_speed'] = $connection_speed; $format[] = '%f'; }
        if ($connection_rtt !== null) { $data['connection_rtt'] = $connection_rtt; $format[] = '%d'; }
        if ($device_type !== null) { $data['device_type'] = $device_type; $format[] = '%s'; }

        $data['visitor_hash'] = $visitor_hash;
        $format[] = '%s';

        // Insert new record
        $result = $wpdb->insert($table, $data, $format);

        if ($result === false) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('VSSV: Database insert failed: ' . $wpdb->last_error);
            }
            wp_send_json_error(['message' => 'Database error', 'error' => $wpdb->last_error]);
        }

        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('VSSV: Successfully inserted vitals, ID: ' . $wpdb->insert_id);
        }

        wp_send_json_success(['message' => 'Collected', 'id' => $wpdb->insert_id]);
    }

    /**
     * Get aggregated Web Vitals from database
     */
    public function get_web_vitals_stats($days = 7) {
        global $wpdb;
        $table = $wpdb->prefix . 'vssv_web_vitals';

        // Get total count
        $total = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM $table WHERE created_at > DATE_SUB(NOW(), INTERVAL %d DAY)",
            $days
        ));

        if (!$total) {
            return [
                'count' => 0,
                'percentiles' => null,
                'scores' => null,
            ];
        }

        // Calculate P75 for each metric
        $metrics = ['lcp', 'fid', 'cls', 'ttfb', 'fcp', 'dom_load', 'page_load'];
        $percentiles = [];

        foreach ($metrics as $metric) {
            $values = $wpdb->get_col($wpdb->prepare(
                "SELECT $metric FROM $table
                 WHERE $metric IS NOT NULL
                 AND created_at > DATE_SUB(NOW(), INTERVAL %d DAY)
                 ORDER BY $metric ASC",
                $days
            ));

            if (count($values) > 0) {
                $index = (int) ceil(0.75 * count($values)) - 1;
                $percentiles[$metric . '_p75'] = floatval($values[max(0, $index)]);
            } else {
                $percentiles[$metric . '_p75'] = null;
            }
        }

        // Calculate scores based on Web Vitals thresholds
        $thresholds = [
            'lcp' => ['good' => 2500, 'poor' => 4000],
            'fid' => ['good' => 100, 'poor' => 300],
            'cls' => ['good' => 0.1, 'poor' => 0.25],
            'ttfb' => ['good' => 800, 'poor' => 1800],
            'fcp' => ['good' => 1800, 'poor' => 3000],
        ];

        $scores = [];
        foreach ($thresholds as $metric => $limits) {
            $value = $percentiles[$metric . '_p75'];
            if ($value === null) {
                $scores[$metric] = 'unknown';
            } elseif ($value <= $limits['good']) {
                $scores[$metric] = 'good';
            } elseif ($value <= $limits['poor']) {
                $scores[$metric] = 'needs-improvement';
            } else {
                $scores[$metric] = 'poor';
            }
        }

        // Get page breakdown
        $pages = $wpdb->get_results($wpdb->prepare(
            "SELECT page_url,
                    COUNT(*) as count,
                    AVG(lcp) as avg_lcp,
                    AVG(fcp) as avg_fcp
             FROM $table
             WHERE created_at > DATE_SUB(NOW(), INTERVAL %d DAY)
             GROUP BY page_url
             ORDER BY count DESC
             LIMIT 20",
            $days
        ), ARRAY_A);

        // Get connection breakdown
        $connections = $wpdb->get_results($wpdb->prepare(
            "SELECT connection_type, COUNT(*) as count
             FROM $table
             WHERE connection_type IS NOT NULL
             AND created_at > DATE_SUB(NOW(), INTERVAL %d DAY)
             GROUP BY connection_type",
            $days
        ), ARRAY_A);

        $connection_breakdown = [];
        foreach ($connections as $conn) {
            $connection_breakdown[$conn['connection_type']] = intval($conn['count']);
        }

        return [
            'count' => intval($total),
            'percentiles' => $percentiles,
            'scores' => $scores,
            'pages' => $pages,
            'connections' => $connection_breakdown,
        ];
    }

    /**
     * AJAX handler for getting performance data
     */
    public function ajax_get_performance_data() {
        check_ajax_referer('vssv_ajax_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }

        // Get performance data from cache and metrics
        $data = [
            'score' => $this->calculate_performance_score(),
            'metrics' => get_option('vssv_performance_factors', []),
            'cache_hit_rate' => get_transient('vssv_cache_hit_rate'),
            'avg_load_time' => get_transient('vssv_avg_load_time'),
            'recommendations' => $this->get_recommendations()
        ];

        wp_send_json_success($data);
    }

    /**
     * AJAX handler for quick actions
     */
    public function ajax_quick_action() {
        check_ajax_referer('vssv_ajax_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }

        $action = isset($_POST['quick_action']) ? sanitize_text_field($_POST['quick_action']) : '';

        if (empty($action)) {
            wp_send_json_error(['message' => 'No action specified']);
        }

        // Execute the requested action
        switch ($action) {
            case 'optimize-images':
                // Trigger image optimization
                do_action('vssv_optimize_images');
                wp_send_json_success(['message' => 'Image optimization started']);
                break;

            case 'minify-assets':
                // Trigger asset minification
                do_action('vssv_minify_assets');
                wp_send_json_success(['message' => 'Asset minification completed']);
                break;

            case 'generate-critical-css':
                // Trigger critical CSS generation
                do_action('vssv_generate_critical_css');
                wp_send_json_success(['message' => 'Critical CSS generation started']);
                break;

            case 'optimize-database':
                // Clean up database
                $this->optimize_database();
                wp_send_json_success(['message' => 'Database optimization completed']);
                break;

            case 'preload-cache':
                // Preload cache
                do_action('vssv_preload_cache');
                wp_send_json_success(['message' => 'Cache preloading started']);
                break;

            case 'test-performance':
                // Run performance test
                $results = $this->run_performance_test();
                wp_send_json_success(['message' => 'Performance test completed', 'results' => $results]);
                break;

            default:
                wp_send_json_error(['message' => 'Unknown action: ' . esc_attr($action)]);
        }
    }

    /**
     * Optimize database
     */
    private function optimize_database() {
        global $wpdb;

        // Delete post revisions
        $wpdb->query( $wpdb->prepare( "DELETE FROM {$wpdb->posts} WHERE post_type = %s", 'revision' ) );

        // Clean expired transients
        delete_expired_transients(true);

        // Optimize tables
        $tables = $wpdb->get_col("SHOW TABLES");
        foreach ($tables as $table) {
            $wpdb->query( $wpdb->prepare( "OPTIMIZE TABLE %s", $table ) );
        }
    }

    /**
     * Run a performance test
     */
    private function run_performance_test() {
        global $wpdb;

        $start_time = microtime(true);

        // Gather actual performance metrics
        $metrics = [
            'score' => $this->calculate_performance_score(),
            'load_time' => get_transient('vssv_avg_load_time') ?: 0,
            'cache_hit_rate' => get_transient('vssv_cache_hit_rate') ?: 0,
            'db_queries' => get_transient('vssv_avg_queries') ?: 0,
            'memory_usage' => memory_get_peak_usage(true),
            'memory_limit' => wp_convert_hr_to_bytes(WP_MEMORY_LIMIT),
            'cache_stats' => get_option('vssv_cache_stats', ['hits' => 0, 'misses' => 0, 'writes' => 0, 'deletes' => 0]),
            'php_version' => PHP_VERSION,
            'wp_version' => get_bloginfo('version'),
            'mysql_version' => $wpdb->db_version()
        ];

        // Get slowest pages from last 24 hours of data
        $slowest_pages = $wpdb->get_results(
            "SELECT page_url, AVG(metric_value) as avg_load_time, COUNT(*) as occurrences
             FROM {$wpdb->prefix}vssv_metrics
             WHERE metric_type = 'load_time'
             AND timestamp >= DATE_SUB(NOW(), INTERVAL 24 HOUR)
             AND page_url IS NOT NULL
             AND page_url != ''
             GROUP BY page_url
             ORDER BY avg_load_time DESC
             LIMIT 10",
            ARRAY_A
        );

        $metrics['slowest_pages'] = $slowest_pages ?: [];

        $end_time = microtime(true);
        $duration = $end_time - $start_time;

        // Store test results
        update_option('vssv_last_performance_test', [
            'timestamp' => current_time('mysql'),
            'duration' => $duration,
            'metrics' => $metrics
        ]);

        return [
            'duration' => round($duration, 4),
            'status' => 'completed',
            'metrics' => $metrics,
            'timestamp' => current_time('mysql')
        ];
    }

    /**
     * AJAX handler for getting dashboard data (for chart and metrics)
     */
    public function ajax_get_dashboard_data() {
        check_ajax_referer('vssv_ajax_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }

        global $wpdb;

        // Get recent metrics from database
        $metrics = $wpdb->get_results(
            "SELECT metric_type, metric_value, timestamp FROM {$wpdb->prefix}vssv_metrics
             ORDER BY timestamp DESC LIMIT 100",
            ARRAY_A
        );

        // Group metrics by day for chart
        $chart_data = [];
        if ($metrics) {
            foreach ($metrics as $metric) {
                $date = date('Y-m-d', strtotime($metric['timestamp']));
                if (!isset($chart_data[$date])) {
                    $chart_data[$date] = [];
                }
                $chart_data[$date][$metric['metric_type']][] = $metric['metric_value'];
            }
        }

        // Calculate averages for chart
        $chart_labels = [];
        $chart_values = [];
        foreach ($chart_data as $date => $data) {
            $chart_labels[] = $date;
            // Average score across all metrics for this day
            $score = isset($data['score']) ? array_sum($data['score']) / count($data['score']) : 0;
            $chart_values[] = round($score, 2);
        }

        wp_send_json_success([
            'chart_data' => [
                'labels' => array_reverse($chart_labels),
                'data' => array_reverse($chart_values)
            ],
            'current_score' => $this->calculate_performance_score(),
            'metrics' => get_option('vssv_performance_factors', []),
            'cache_stats' => get_option('vssv_cache_stats', ['hits' => 0, 'misses' => 0, 'writes' => 0, 'deletes' => 0])
        ]);
    }

    /**
     * AJAX handler for performance history data
     */
    public function ajax_get_performance_history() {
        check_ajax_referer('vssv_ajax_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }

        global $wpdb;

        // Get parameters
        $days = isset($_POST['days']) ? intval($_POST['days']) : 7;
        $page = isset($_POST['page']) ? max(1, intval($_POST['page'])) : 1;
        $per_page = 10;
        $offset = ($page - 1) * $per_page;

        // Validate days parameter
        $days = in_array($days, [7, 14, 30]) ? $days : 7;

        // Get metrics from the database grouped by date
        $metrics_data = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT
                    DATE(timestamp) as date,
                    metric_type,
                    AVG(CAST(metric_value AS DECIMAL(10,2))) as avg_value,
                    MIN(CAST(metric_value AS DECIMAL(10,2))) as min_value,
                    MAX(CAST(metric_value AS DECIMAL(10,2))) as max_value
                 FROM {$wpdb->prefix}vssv_metrics
                 WHERE timestamp >= DATE_SUB(NOW(), INTERVAL %d DAY)
                 GROUP BY DATE(timestamp), metric_type
                 ORDER BY date DESC",
                $days
            ),
            ARRAY_A
        );

        // Get slowest pages
        $slowest_pages = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT
                    page_url as url,
                    AVG(CAST(metric_value AS DECIMAL(10,2))) as avg_load_time,
                    COUNT(*) as occurrences
                 FROM {$wpdb->prefix}vssv_metrics
                 WHERE timestamp >= DATE_SUB(NOW(), INTERVAL %d DAY)
                 AND metric_type = 'load_time'
                 AND page_url IS NOT NULL
                 AND page_url != ''
                 GROUP BY page_url
                 ORDER BY avg_load_time DESC
                 LIMIT 10",
                $days
            ),
            ARRAY_A
        );

        // Get all detailed metrics for the table
        $all_metrics = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT
                    DATE(timestamp) as date,
                    metric_type,
                    AVG(CAST(metric_value AS DECIMAL(10,2))) as avg_value
                 FROM {$wpdb->prefix}vssv_metrics
                 WHERE timestamp >= DATE_SUB(NOW(), INTERVAL %d DAY)
                 GROUP BY DATE(timestamp), metric_type
                 ORDER BY date DESC
                 LIMIT %d OFFSET %d",
                $days,
                $per_page,
                $offset
            ),
            ARRAY_A
        );

        // Process metrics into chart data
        $chart_data = [];
        $stats_data = [];

        if ($metrics_data) {
            foreach ($metrics_data as $metric) {
                $date = $metric['date'];
                if (!isset($chart_data[$date])) {
                    $chart_data[$date] = [];
                }
                $chart_data[$date][$metric['metric_type']] = floatval($metric['avg_value']);
            }

            // Calculate averages
            $scores = [];
            $load_times = [];
            $cache_rates = [];
            $queries = [];

            foreach ($chart_data as $date => $metrics) {
                if (isset($metrics['score'])) $scores[] = $metrics['score'];
                if (isset($metrics['load_time'])) $load_times[] = $metrics['load_time'];
                if (isset($metrics['cache_hit_rate'])) $cache_rates[] = $metrics['cache_hit_rate'];
                if (isset($metrics['db_queries'])) $queries[] = $metrics['db_queries'];
            }

            $avg_score = count($scores) > 0 ? array_sum($scores) / count($scores) : 0;
            $avg_load_time = count($load_times) > 0 ? array_sum($load_times) / count($load_times) : 0;
            $avg_cache_rate = count($cache_rates) > 0 ? array_sum($cache_rates) / count($cache_rates) : 0;
            $avg_queries = count($queries) > 0 ? array_sum($queries) / count($queries) : 0;

            // Calculate trends (compare first half vs second half of period)
            $first_half = array_slice($scores, count($scores) / 2);
            $second_half = array_slice($scores, 0, count($scores) / 2);

            $first_half_avg = count($first_half) > 0 ? array_sum($first_half) / count($first_half) : $avg_score;
            $second_half_avg = count($second_half) > 0 ? array_sum($second_half) / count($second_half) : $avg_score;

            $score_trend = ($second_half_avg > $first_half_avg * 1.02) ? 'improving' : (($second_half_avg < $first_half_avg * 0.98) ? 'declining' : 'stable');

            // Similar for load time (lower is better, so reverse the logic)
            $load_time_trend = ($second_half_avg < $first_half_avg * 0.98) ? 'improving' : (($second_half_avg > $first_half_avg * 1.02) ? 'declining' : 'stable');

            $stats_data = [
                'avg_score' => round($avg_score, 2),
                'score_trend' => $score_trend,
                'avg_load_time' => round($avg_load_time, 2),
                'load_time_trend' => $load_time_trend,
                'avg_cache_rate' => round($avg_cache_rate, 2),
                'avg_queries' => round($avg_queries, 2),
                'avg_memory' => '0MB'
            ];
        }

        // Build chart labels and data
        $chart_labels = array_keys($chart_data);
        $chart_labels = array_reverse($chart_labels);
        $chart_scores = [];
        $chart_load_times = [];

        foreach ($chart_labels as $label) {
            $chart_scores[] = isset($chart_data[$label]['score']) ? floatval($chart_data[$label]['score']) : 0;
            $chart_load_times[] = isset($chart_data[$label]['load_time']) ? floatval($chart_data[$label]['load_time']) : 0;
        }

        // Format detailed metrics for table
        $detailed_metrics = [];
        if ($all_metrics) {
            // Group by date
            $metrics_by_date = [];
            foreach ($all_metrics as $metric) {
                $date = $metric['date'];
                if (!isset($metrics_by_date[$date])) {
                    $metrics_by_date[$date] = [];
                }
                $metrics_by_date[$date][$metric['metric_type']] = floatval($metric['avg_value']);
            }

            foreach ($metrics_by_date as $date => $metrics) {
                $detailed_metrics[] = [
                    'date' => $date,
                    'score' => isset($metrics['score']) ? round($metrics['score'], 0) : 0,
                    'load_time' => isset($metrics['load_time']) ? round($metrics['load_time'], 2) : 0,
                    'cache_rate' => isset($metrics['cache_hit_rate']) ? round($metrics['cache_hit_rate'], 0) : 0,
                    'queries' => isset($metrics['db_queries']) ? round($metrics['db_queries'], 0) : 0,
                    'memory' => '0MB'
                ];
            }
        }

        wp_send_json_success([
            'chart_data' => [
                'labels' => $chart_labels,
                'scores' => $chart_scores,
                'load_times' => $chart_load_times
            ],
            'statistics' => $stats_data,
            'slowest_pages' => $slowest_pages,
            'detailed_metrics' => $detailed_metrics,
            'pagination' => [
                'current_page' => $page,
                'total_pages' => ceil(count($all_metrics) / $per_page)
            ]
        ]);
    }

    /**
     * Register REST API routes
     */
    public function register_rest_routes() {
        register_rest_route('vssv/v1', '/performance', [
            'methods' => 'GET',
            'callback' => [$this, 'rest_get_performance'],
            'permission_callback' => function() {
                return current_user_can('manage_options');
            }
        ]);

        register_rest_route('vssv/v1', '/optimize', [
            'methods' => 'POST',
            'callback' => [$this, 'rest_run_optimization'],
            'permission_callback' => function() {
                return current_user_can('manage_options');
            }
        ]);
    }

    /**
     * REST API: Get performance data
     */
    public function rest_get_performance($request) {
        return new WP_REST_Response([
            'score' => $this->calculate_performance_score(),
            'metrics' => get_option('vssv_performance_factors', []),
            'recommendations' => $this->get_recommendations()
        ]);
    }

    /**
     * REST API: Run optimization
     */
    public function rest_run_optimization($request) {
        $results = [];

        foreach ($this->modules as $name => $module) {
            if (method_exists($module, 'optimize')) {
                $results[$name] = $module->optimize();
            }
        }

        return new WP_REST_Response([
            'success' => true,
            'results' => $results
        ]);
    }

    /**
     * Get performance recommendations
     */
    private function get_recommendations() {
        $recommendations = [];

        // Check each module for recommendations
        foreach ($this->modules as $module) {
            if (method_exists($module, 'get_recommendations')) {
                $recommendations = array_merge(
                    $recommendations,
                    $module->get_recommendations()
                );
            }
        }

        return $recommendations;
    }

    /**
     * Log optimization activity
     */
    private function log_optimization($type, $details) {
        global $wpdb;

        $wpdb->insert(
            $wpdb->prefix . 'vssv_optimization_log',
            [
                'optimization_type' => $type,
                'after_value' => $details,
                'timestamp' => current_time('mysql')
            ]
        );
    }

    /**
     * Show requirements notice
     */
    public function show_requirements_notice() {
        $requirements = get_option('vssv_requirements_missing', []);

        if (!empty($requirements)) {
            echo '<div class="notice notice-warning">';
            echo '<p><strong>VS Site Vector:</strong> Some requirements are not met:</p>';
            echo '<ul style="list-style: disc; margin-left: 20px;">';
            foreach ($requirements as $req) {
                echo '<li>' . esc_html($req) . '</li>';
            }
            echo '</ul>';
            echo '</div>';
        }
    }

    /**
     * Cleanup expired transients
     */
    private function cleanup_expired_transients() {
        if (rand(1, 100) > 5) {
            return; // Only run 5% of the time
        }

        global $wpdb;

        $wpdb->query(
            $wpdb->prepare(
                "DELETE FROM {$wpdb->options}
                WHERE option_name LIKE %s
                AND option_value < %d",
                $wpdb->esc_like('_transient_timeout_') . '%',
                time()
            )
        );
    }
}

// Initialize plugin
add_action('plugins_loaded', function() {
    VS_Site_Vector::get_instance();
}, 0);

// Global function for external access
function vssv() {
    return VS_Site_Vector::get_instance();
}